# run_skeleton_task.py (Executable with Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Predefined skills only!

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Info] Object positions loaded:", positions)

        # === Begin Exploration Phase ===
        # Based on feedback, we need to discover the missing predicate by trying available exploration skills.
        # We assume we have access to the following skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Try all available exploration actions to see which predicates are required/missing
        # Exploration plan: Try to 'pick', 'move', 'push', 'pull', etc., and print status/results

        # We'll iterate through objects and places from the observation/positions loaded
        objects = [k for k in positions.keys() if 'drawer' not in k.lower()]
        drawers = [k for k in positions.keys() if 'drawer' in k.lower()]
        locations = list({tuple(v) for v in positions.values()})
        # For locations, we'll keep the dictionary keys only, for labeling and skill-calling
        location_names = set()
        for name, pos in positions.items():
            if isinstance(pos, (tuple, list)) and len(pos) == 3:
                location_names.add(name)
        location_names = list(location_names)

        # We'll simulate a phase of trying to execute all relevant exploration skills and catch exceptions
        # The plan is: For each object and location, try the major skills, and log any failure/success.
        for obj_name in objects:
            obj_pos = positions[obj_name]
            for loc_name in location_names:
                try:
                    print(f"[Exploration] Trying to execute_pick on '{obj_name}' at '{loc_name}'...")
                    obs, reward, done = execute_pick(env, task, target=obj_name, location=loc_name)
                    print(f"[Exploration] Success: execute_pick for {obj_name} at {loc_name}")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj_name} at {loc_name}: {e}")

                try:
                    print(f"[Exploration] Trying to execute_place on '{obj_name}' at '{loc_name}'...")
                    obs, reward, done = execute_place(env, task, target=obj_name, location=loc_name)
                    print(f"[Exploration] Success: execute_place for {obj_name} at {loc_name}")
                except Exception as e:
                    print(f"[Exploration] execute_place failed for {obj_name} at {loc_name}: {e}")

        # Try to push and pull for drawers
        for drawer in drawers:
            print(f"[Exploration] Testing drawer: {drawer}")
            for loc_name in location_names:
                try:
                    print(f"[Exploration] Trying to execute_push on '{drawer}' at '{loc_name}'...")
                    obs, reward, done = execute_push(env, task, drawer=drawer, location=loc_name)
                    print(f"[Exploration] Success: execute_push for {drawer} at {loc_name}")
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {drawer} at {loc_name}: {e}")

                try:
                    print(f"[Exploration] Trying to execute_pull on '{drawer}' at '{loc_name}'...")
                    obs, reward, done = execute_pull(env, task, drawer=drawer, location=loc_name)
                    print(f"[Exploration] Success: execute_pull for {drawer} at {loc_name}")
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {drawer} at {loc_name}: {e}")

        # Try sweep and gripper skill for all available objects/locations
        for obj_name in objects:
            for loc_name in location_names:
                try:
                    print(f"[Exploration] Trying to execute_sweep on '{obj_name}' at '{loc_name}'...")
                    obs, reward, done = execute_sweep(env, task, target=obj_name, location=loc_name)
                    print(f"[Exploration] Success: execute_sweep for {obj_name} at {loc_name}")
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj_name} at {loc_name}: {e}")

        try:
            print("[Exploration] Trying to execute_gripper...")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] Success: execute_gripper")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # Try go/move between all pairs of locations
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying to execute_go from '{from_loc}' to '{to_loc}'...")
                        obs, reward, done = execute_go(env, task, from_location=from_loc, to_location=to_loc)
                        print(f"[Exploration] Success: execute_go from {from_loc} to {to_loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        print("===== Exploration Complete: Check the above logs to see if any skill failed due to a missing predicate (e.g., not identified, temperature-known, weight-known, etc.).")
        print("If a skill failed with a missing predicate error or precondition, that is likely the missing predicate!")

        # After exploration, you would typically update the planning model to include/learn the missing predicate
        # and then retry the task execution once the issue is identified.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()