# run_skeleton_task.py (Filled with Exploration Phase and Plan Execution)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *  # pre-existing skill functions only
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton, now with exploration phase for missing predicates.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # For demonstration, print all positions
        print("[Exploration] Object positions in env:", positions)

        # === PHASE 1: Exploration for Missing Predicates ===
        print("[Exploration] Begin exploration to determine missing predicates...")

        # Note: Exploration phase will attempt to call each available skill on the objects to see if anything is missing

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For exploration, we try to 'touch' all relevant environment facets.
        # Let's enumerate through typical pickable objects and locations
        # Assume positions dictionary: object name -> (type, position)
        # We'll group objects by type (object, drawer, handle, location, etc.)

        # Identify instance names to try skills; fallback to generic if not known

        # Attempt each skill and catch exceptions to see constraint violations
        exploration_results = {}
        for skill_name in available_skills:
            try:
                fn = globals()[skill_name]
                print(f"[Exploration] Trying skill: {skill_name}")

                # For each skill, call it with plausible dummy arguments
                if skill_name == 'execute_go':
                    # Try moving from one known location to another
                    locations = [k for k, v in positions.items() if 'location' in k or 'pose' in k]
                    if len(locations) >= 2:
                        obs, reward, done = fn(
                            env, task, from_location=locations[0], to_location=locations[1]
                        )
                    else:
                        print(f"[Exploration:{skill_name}] Not enough locations; skipped.")
                elif skill_name == 'execute_pick':
                    # Try picking an object from its known position
                    pickable_objs = [k for k in positions.keys() if 'object' in k or 'ball' in k]
                    # Try first pickable
                    if pickable_objs:
                        obj = pickable_objs[0]
                        loc_candidates = [k for k in positions.keys() if 'location' in k or 'pose' in k]
                        loc = loc_candidates[0] if loc_candidates else 'default_location'
                        obs, reward, done = fn(
                            env, task, obj=obj, location=loc
                        )
                    else:
                        print(f"[Exploration:{skill_name}] No pickable objects; skipped.")
                elif skill_name == 'execute_place':
                    # Place the object, suppose the drawer is available
                    drawers = [k for k in positions if 'drawer' in k]
                    pickable_objs = [k for k in positions if 'object' in k or 'ball' in k]
                    loc_candidates = [k for k in positions.keys() if 'location' in k or 'pose' in k]
                    if drawers and pickable_objs and loc_candidates:
                        obj = pickable_objs[0]
                        drawer = drawers[0]
                        loc = loc_candidates[0]
                        obs, reward, done = fn(
                            env, task, obj=obj, drawer=drawer, location=loc
                        )
                    else:
                        print(f"[Exploration:{skill_name}] No drawer/placeable object; skipped.")
                elif skill_name == 'execute_pull':
                    # Pull on a handle/drawer
                    handles = [k for k in positions if 'handle' in k]
                    drawers = [k for k in positions if 'drawer' in k]
                    loc_candidates = [k for k in positions if 'location' in k or 'pose' in k]
                    if handles and drawers and loc_candidates:
                        drawer = drawers[0]
                        handle = handles[0]
                        loc = loc_candidates[0]
                        obs, reward, done = fn(
                            env, task, drawer=drawer, handle=handle, location=loc
                        )
                    else:
                        print(f"[Exploration:{skill_name}] No handle/drawer available; skipped.")
                elif skill_name == 'execute_push':
                    # Push a drawer, use location if needed
                    drawers = [k for k in positions if 'drawer' in k]
                    loc_candidates = [k for k in positions.keys() if 'location' in k or 'pose' in k]
                    if drawers and loc_candidates:
                        drawer = drawers[0]
                        loc = loc_candidates[0]
                        obs, reward, done = fn(
                            env, task, drawer=drawer, location=loc
                        )
                    else:
                        print(f"[Exploration:{skill_name}] No drawer for push; skipped.")
                elif skill_name == 'execute_sweep':
                    # Sweep near an object
                    pickable_objs = [k for k in positions.keys() if 'object' in k or 'ball' in k]
                    loc_candidates = [k for k in positions.keys() if 'location' in k or 'pose' in k]
                    if pickable_objs and loc_candidates:
                        obj = pickable_objs[0]
                        loc = loc_candidates[0]
                        obs, reward, done = fn(
                            env, task, obj=obj, location=loc
                        )
                    else:
                        print(f"[Exploration:{skill_name}] No object/location for sweep; skipped.")
                elif skill_name == 'execute_rotate':
                    # Rotate an object, if available
                    rotatable_objs = [k for k in positions.keys() if 'object' in k or 'knob' in k]
                    if rotatable_objs:
                        obj = rotatable_objs[0]
                        obs, reward, done = fn(env, task, obj=obj)
                    else:
                        print(f"[Exploration:{skill_name}] No object to rotate; skipped.")
                elif skill_name == 'execute_gripper':
                    # Gripper open/close, no args
                    obs, reward, done = fn(env, task)
                else:
                    print(f"[Exploration:{skill_name}] No handler; skipped.")
                
                print(f"[Exploration:{skill_name}] Executed without exception.")
                exploration_results[skill_name] = 'success'
            except Exception as e:
                print(f"[Exploration:{skill_name}] Exception during execution: {e}")
                exploration_results[skill_name] = f'failed:{str(e)}'
        
        print("[Exploration] Exploration phase results:", exploration_results)
        print("[Exploration] Exploration phase complete. If any skill failed due to a missing predicate, inspect error messages above.")

        # === PHASE 2: Plan Execution ===

        print("[Plan] Beginning plan execution...")
        # Your domain-specific oracle plan goes here.
        # Example:
        #    (1) Move to object
        #    (2) Pick object
        #    (3) Move to drawer
        #    (4) Open drawer (pull)
        #    (5) Place object in drawer
        #    (6) Push drawer to close

        # CAUTION: These calls must match `skill_code` signatures for your RLBench configuration.
        # Example pseudocode plan execution (adjust argument names/ordering per your environment):

        try:
            # 1) Move to object location
            object_locs = [k for k in positions if 'object' in k or 'ball' in k]
            locations = [k for k in positions if 'location' in k or 'pose' in k]
            drawers = [k for k in positions if 'drawer' in k]

            if not object_locs or not locations or not drawers:
                print("[Plan] Not enough known objects/locations/drawers in environment.")
            else:
                obj = object_locs[0]
                from_loc = locations[0]
                to_loc = locations[1] if len(locations) > 1 else locations[0]
                drawer = drawers[0]

                print(f"[Plan] Move to {from_loc}")
                obs, reward, done = execute_go(env, task, from_location=from_loc, to_location=to_loc)

                print(f"[Plan] Pick {obj}")
                obs, reward, done = execute_pick(env, task, obj=obj, location=to_loc)

                print(f"[Plan] Move to drawer {drawer} at {to_loc}")
                obs, reward, done = execute_go(env, task, from_location=to_loc, to_location=drawer)

                # For handles: if handles available, otherwise try direct pull if permitted
                handles = [k for k in positions if 'handle' in k]
                if handles:
                    handle = handles[0]
                    print(f"[Plan] Pull drawer {drawer} with handle {handle}")
                    obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location=drawer)
                else:
                    print(f"[Plan] No handle for pulling drawer {drawer}; skipping pull step.")

                print(f"[Plan] Place {obj} in {drawer}")
                obs, reward, done = execute_place(env, task, obj=obj, drawer=drawer, location=drawer)

                print(f"[Plan] Push drawer {drawer} closed")
                obs, reward, done = execute_push(env, task, drawer=drawer, location=drawer)
            
                print("[Plan] Task plan executed successfully.")

        except Exception as e:
            print(f"[Plan] Exception during execution: {e}")

        print("===== Exploration and Plan Execution Complete =====")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()