# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all provided primitive skills; do not redefine
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Should be dictionary: object name -> pose/position tuple
        try:
            positions = get_object_positions()
        except Exception as e:
            print("[Error] Could not get object positions:", e)
            positions = dict()  # fallback to empty

        # === Exploration: Attempt to discover missing predicate/capability ===
        # As per Feedback: (robot-at trash) was mentioned as an unexpected fact/predicate.
        # Therefore, systematically try to move the robot to location 'trash' and see state.

        TRASH_LOCATION = 'trash'
        # You may need a list of all possible locations; here, try to get robot's current location.
        robot_location = None

        # Attempt to infer initial robot location from positions or obs
        # For this skeleton, try to find from obs or fallback to common names
        try:
            # If obs dict provides robot's location or pose explicitly
            if isinstance(obs, dict):
                if 'robot' in obs and 'location' in obs['robot']:
                    robot_location = obs['robot']['location']
                elif 'robot_at' in obs:
                    robot_location = obs['robot_at']
                elif 'robot-location' in obs:
                    robot_location = obs['robot-location']
            # Fallback: Try default starting pose names
            if robot_location is None:
                robot_location = 'ready-pose'
            print(f"[Exploration] Current robot location: {robot_location}")
        except Exception as e:
            robot_location = 'ready-pose'
            print(f"[Exploration] Could not determine robot location: {e}")

        # Use execute_go (move) skill to go to 'trash' location
        # For exploration, we record the observation before and after move
        print(f"[Exploration] Moving robot from {robot_location} to '{TRASH_LOCATION}' to check for missing predicate.")

        try:
            # Call the execute_go skill to move
            obs_before, _, done = execute_go(env, task, from_loc=robot_location, to_loc=TRASH_LOCATION)
            print(f"[Exploration] execute_go: from={robot_location} to={TRASH_LOCATION}")
            # After move, get the new observation
            current_obs = task.get_observation()
            # Examine current observation or state for the predicate
            # (In real systems you might inspect task's state or any debug information here)
            print(f"[Exploration] Observation after moving to '{TRASH_LOCATION}': {current_obs}")
        except Exception as e:
            print(f"[Exploration] Failed to move to '{TRASH_LOCATION}': {e}")
            # Gracefully handle or recover
            current_obs = None

        # You may wish to try other known skills or locations as needed in response to feedback
        # For now, this phase simply demonstrates execution and observation capturing

        print("[Exploration] Exploration phase complete. Please check the log/observation for evidence of the missing predicate.")

        # ---
        # Normal plan/task logic would go here (e.g., pick/place sequences), which is not
        # specified due to the focus on exploration for the missing predicate.
        # ---

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
