# run_skeleton_task.py (Completed with Exploration Logic for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # === Exploration Phase: Identify Missing Predicate ===
        # As per feedback, a timeout/plan failure suggests a missing predicate.
        # The exploration domain's actions are designed to learn/identify these.
        # We'll execute each available exploration action for possible predicates, then log which succeeded.
        exploration_skills = ['execute_go', 'execute_pick', 'execute_pull']
        # We'll store which predicates we discover successfully.
        discovered_predicates = {}

        # Assume positions contains mappings for robot and objects
        # E.g., positions = { 'robot': ... , 'object1': ... , 'drawer1': ... }
        # We'll look for objects, drawers, and locations in this dict
        robot_position = positions.get('robot', None)
        location_keys = []
        object_keys = []
        drawer_keys = []
        handle_keys = []
        for k, v in positions.items():
            if 'drawer' in k:
                drawer_keys.append(k)
            elif 'handle' in k:
                handle_keys.append(k)
            elif 'object' in k or k.startswith('obj'):
                object_keys.append(k)
            elif 'loc' in k or k.startswith('loc') or k.startswith('room'):
                location_keys.append(k)
        # Provide fallback for generic location search
        if not location_keys and robot_position:
            location_keys = [k for k in positions if k != 'robot']

        print("[Exploration] Starting to test for missing predicates/properties...")

        # Try each exploration skill (try-catch in case skill is not available)
        # For each object at each location, try exploration actions
        for obj in object_keys:
            for loc in location_keys:
                try:
                    # Try to 'go' to location and see if object becomes "identified"
                    print(f"[Exploration] Testing identify for {obj} at {loc}")
                    obs, reward, done = None, 0, False
                    try:
                        obs, reward, done = execute_go(env, task, from_location=robot_position, to_location=loc)
                        discovered_predicates.setdefault(obj, []).append('identified')
                    except Exception as e:
                        print(f"[Exploration] execute_go for identify failed: {e}")
                    
                    # Try to pick and see if we get known weight or durability
                    print(f"[Exploration] Testing pick (weight/durability) for {obj} at {loc}")
                    try:
                        obs, reward, done = execute_pick(env, task, object_name=obj, location=loc)
                        discovered_predicates.setdefault(obj, []).append('weight-known')
                        discovered_predicates[obj].append('durability-known')
                    except Exception as e:
                        print(f"[Exploration] execute_pick for weight/durability failed: {e}")

                    # Try to pull and see if lock-known becomes available
                    print(f"[Exploration] Testing pull (lock-known) for {obj} at {loc}")
                    try:
                        obs, reward, done = execute_pull(env, task, object_name=obj, location=loc)
                        discovered_predicates.setdefault(obj, []).append('lock-known')
                    except Exception as e:
                        print(f"[Exploration] execute_pull for lock-known failed: {e}")

                except Exception as e:
                    print(f"[Exploration] Error while trying skills on {obj} at {loc}: {e}")

        # Log the discovered predicates per object
        print("[Exploration] Discovered predicates info for objects:")
        for obj, preds in discovered_predicates.items():
            print(f"  {obj}: {set(preds)}")

        # === Example Oracle Plan Execution ===
        # After exploration, run normal planned actions as would be in the oracle plan.
        # (Placeholder - would be dynamically generated based on proper planning)
        print("[Task] Executing main oracle plan (example, to be replaced with real plan)...")
        # As an example, using the available skill names, you might execute:
        # - Go to a location
        # - Pick up an object
        # - Place that object or do other manipulation as needed
        # Adjust argument names as needed for your skill functions.
        try:
            if object_keys and location_keys:
                target_obj = object_keys[0]
                target_loc = location_keys[0]

                print(f"[Task] Moving robot to {target_loc}")
                obs, reward, done = execute_go(env, task, from_location=robot_position, to_location=target_loc)

                print(f"[Task] Picking {target_obj} at {target_loc}")
                obs, reward, done = execute_pick(env, task, object_name=target_obj, location=target_loc)

                # Example of a place or push, adjust as appropriate
                if drawer_keys:
                    drawer = drawer_keys[0]
                    print(f"[Task] Placing {target_obj} into {drawer} at {target_loc}")
                    obs, reward, done = execute_place(env, task, object_name=target_obj, drawer_name=drawer, location=target_loc)
        except Exception as plan_exc:
            print(f"[Task] Error during plan execution: {plan_exc}")

        print("===== Task complete. =====")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
