# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # You don't need to redefine primitives like move, pick, place

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # {'drawer': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase to Discover Missing Predicate ===
        #
        # Based on feedback and the domain, planner is likely missing knowledge 
        # about an object or object state that blocks planning.
        #
        # We use the exploration knowledge skills (notably, execute_pull) to 
        # check if a predicate like 'lock-known' is missing for a drawer.
        #
        # For this, try using exploration skills to update/remove missing predicate, 
        # then reattempt normal plan.
        #
        # We assume for demonstration there's a 'drawer' and a 'handle' (you could generalize).

        # Example names, update to match your env!
        drawer_name = 'drawer'
        handle_name = 'handle'
        robot_current_location = None
        # Try to get these positions; if missing, exception will be handled later.
        try:
            drawer_pos = positions[drawer_name]
            handle_pos = positions[handle_name]
        except Exception as e:
            print("[Exploration] Could not retrieve drawer or handle from positions:", e)
            print("[Exploration] Printing all available positions:", positions)
            # Cannot proceed if these are missing
            return

        # 1. Move to handle location, if not already there
        try:
            # If we know where robot is, can check; else just go
            # Assume 'ready_pose' is home for the robot, else pick any known loc
            robot_current_location = 'ready_pose' if 'ready_pose' in positions else list(positions.keys())[0]
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_current_location,
                to_location=handle_pos,
                timeout=10.0
            )
            robot_current_location = handle_pos
        except Exception as e:
            print("[Exploration] Failed to move to handle position:", e)

        # 2. Pick the handle, if required by the plan
        try:
            obs, reward, done = execute_pick_handle(
                env,
                task,
                object_name=handle_name,
                location=handle_pos,
                timeout=10.0
            )
        except Exception as e:
            print("[Exploration] Failed to pick handle:", e)

        # 3. Check if we need to discover lock-known
        # Try to pull the drawer (exploration domain: this reveals lock status)
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name,
                handle_name,
                handle_pos,
                timeout=10.0
            )
        except Exception as e:
            print("[Exploration] Pull (lock-known discovery) may have failed, ok if known:", e)
        
        # You may want to push back to ready pose for normal plan...
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=handle_pos,
                timeout=10.0
            )
            robot_current_location = 'ready_pose'
        except Exception as e:
            print("[Exploration] Return to ready pose failed (may not be necessary):", e)

        print("[Exploration] Finished. Now attempting original plan (Task Phase).")

        # === Main Task Plan ===
        #
        # Example Oracle Plan (pseudocode, adapt to your specific task plan):
        # For demonstration, we will:
        # - Move to object
        # - Pick object
        # - Move to drawer
        # - Place object
        # - Close drawer (if required)
        #
        # Replace 'object_name' and other references to match your domain.

        object_name = 'object'
        try:
            object_pos = positions[object_name]
        except Exception as e:
            print("[Task] Could not find the main object:", e)
            return

        # a. Move to object location
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_current_location,
                to_location=object_pos,
                timeout=10.0
            )
            robot_current_location = object_pos
        except Exception as e:
            print("[Task] Failed to move to object:", e)
            # Can decide to return or proceed

        # b. Pick the object
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_name,
                location=object_pos,
                timeout=10.0
            )
        except Exception as e:
            print("[Task] Failed to pick object:", e)
            return

        # c. Go to drawer
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_current_location,
                to_location=drawer_pos,
                timeout=10.0
            )
            robot_current_location = drawer_pos
        except Exception as e:
            print("[Task] Failed to move to drawer:", e)
            return

        # d. Open the drawer (if required; use handle if needed)
        # (We assume this was already handled in the exploration phase. Skip.)

        # e. Place the object in the drawer
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_name,
                drawer_name=drawer_name,
                location=drawer_pos,
                timeout=10.0
            )
        except Exception as e:
            print("[Task] Failed to place object in drawer:", e)

        # f. Optionally, close drawer if needed
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_pos,
                timeout=10.0
            )
        except Exception as e:
            print("[Task] Failed to close drawer (push):", e)

        # g. Return to ready pose (optional)
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_pos,
                timeout=10.0
            )
        except Exception as e:
            print("[Task] Return to ready pose failed (end of task):", e)

        print("===== Task completed (exploration and main plan executed) =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()