# run_skeleton_task.py (Exploration + Execution with Safety and Force Calibration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # All primitive skills are imported

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset environment to initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        # === Retrieve live Object Positions and Properties ===
        positions = get_object_positions()
        # For demonstration, print out detected positions
        print("[Info] Object Positions: ", positions)
        # Object names and properties for safety/force calibration
        object_names = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "trash",
            "trash_bin"
        ]
        object_properties = {
            "trash": {"weight": 0.5, "size": 0.2},
            "trash_bin": {"weight": 0.1, "size": 0.05}
        }
        force_calibration = 0.5   # Newtons (base force calibration)
        
        # =============== Exploration Phase ====================
        # Try to discover if we know weight/durability information for objects -- if not, explore!
        # For this demonstration, we'll try to "explore" properties for "trash" and "trash_bin"
        for obj_name in ["trash", "trash_bin"]:
            if obj_name not in positions:
                print(f"[WARN] Object {obj_name} not found in environment. Skipping exploration.")
                continue
            # Suppose the robot needs to acquire weight/durability for this object before plan execution
            obj_pos = positions[obj_name]
            try:
                print(f"[Exploration] Moving to location of {obj_name} for exploration.")
                # Move robot to object location if not already there
                robot_loc = None
                for k, v in positions.items():
                    if k == "robot" or "robot" in k:
                        robot_loc = v
                        break
                # If robot position known and not at object, move to object
                if robot_loc is not None and robot_loc != obj_pos:
                    # Use a 'go' primitive to move between locations -- skill name: execute_go
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_pos=robot_loc,
                        to_pos=obj_pos,
                        max_steps=60,
                        approach_axis='z',  # example axis
                        timeout=10.0
                    )
                # Now, try to "explore" object's weight and/or durability using a safe pick-up (soft grasp)
                # Safety and object-list checks before picking up object
                if obj_name not in object_names:
                    print(f"[ERROR] {obj_name} is not a valid object. Skipping...")
                    continue
                prop_json = object_properties.get(obj_name, {"weight": 0.2, "size": 0.05})
                est_weight = prop_json["weight"]
                est_size = prop_json["size"]
                grasp_force = force_calibration * (est_weight / est_size)
                if grasp_force < 0.01:
                    print(f"[Safety] Calibration result - insufficient force ({grasp_force:.4f}N) to pick up {obj_name}")
                    continue
                print(f"[Exploration] Attempting to pick up {obj_name} with calibrated force {grasp_force:.2f}N for property acquisition")
                # Pick action (simulate force calibration via parameter, if supported in codebase)
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_name=obj_name,
                    target_pos=obj_pos,
                    approach_distance=0.10,
                    max_steps=80,
                    threshold=0.008,
                    approach_axis='z',
                    timeout=7.0,
                    force=grasp_force  # If the skill supports, else ignored
                )
                if not done:
                    print(f"[Exploration] Failed to pick (explore) {obj_name}. Skipping durability check.")
                else:
                    print(f"[Exploration] Picked {obj_name}. Properties (weight/durability) may now be known.")
                # Optionally, can drop/place the object back as part of exploration
            except Exception as e:
                print(f"[EXPLORATION ERROR] ({obj_name}): {e}")
                continue
        
        # ================= Plan Execution Phase =================
        # Normally, a predefined oracle plan would be given, here is a generic demonstration:
        try:
            # Example plan steps after exploration (assume object to pick and place)
            # 1. Move to object 2. Pick safely 3. Move to destination 4. Place safely
            
            plan_target = "trash"
            plan_destination = "trash_bin"
            if plan_target in positions and plan_destination in positions:
                # Move to trash location (if not already there)
                robot_loc = None
                for k, v in positions.items():
                    if k == "robot" or "robot" in k:
                        robot_loc = v
                        break
                if robot_loc is not None and robot_loc != positions[plan_target]:
                    print(f"[Plan] Moving robot to {plan_target} at {positions[plan_target]}")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_pos=robot_loc,
                        to_pos=positions[plan_target],
                        max_steps=60,
                        approach_axis='z',
                        timeout=10.0
                    )
                # Pick up trash safely
                prop_json = object_properties.get(plan_target, {"weight": 0.2, "size": 0.05})
                grasp_force = force_calibration * (prop_json["weight"] / prop_json["size"])
                print(f"[Plan] Picking up {plan_target} with grasp force {grasp_force:.2f}N")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_name=plan_target,
                    target_pos=positions[plan_target],
                    approach_distance=0.10,
                    max_steps=80,
                    threshold=0.008,
                    approach_axis='z',
                    timeout=7.0,
                    force=grasp_force
                )
                if not done:
                    print(f"[Plan] Failed to pick up {plan_target}. Ending task.")
                    return
                # Move to trash_bin location
                print(f"[Plan] Moving to {plan_destination} at {positions[plan_destination]}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_pos=positions[plan_target],
                    to_pos=positions[plan_destination],
                    max_steps=60,
                    approach_axis='z',
                    timeout=10.0
                )
                # Place trash in trash_bin
                print(f"[Plan] Placing {plan_target} into {plan_destination}")
                obs, reward, done = execute_place(
                    env,
                    task,
                    target_name=plan_target,
                    dest_name=plan_destination,
                    dest_pos=positions[plan_destination],
                    approach_distance=0.10,
                    max_steps=60,
                    threshold=0.008,
                    approach_axis='z',
                    timeout=7.0,
                )
                if done:
                    print(f"[Plan] Successfully placed {plan_target} in {plan_destination}. Task complete.")
                else:
                    print(f"[Plan] Failed to place {plan_target} in {plan_destination}.")
            else:
                print(f"[Plan] Required objects not found for plan execution.")
        except Exception as e:
            print(f"[TASK EXECUTION ERROR]: {e}")
        
        # Optionally, additional plan steps can be inserted here.
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
