# run_skeleton_task.py (Executable Task Code with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Predefined: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation with integrated exploration for missing predicate detection.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Optional: Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions and Semantic Mapping ===
        try:
            positions = get_object_positions()
        except Exception as e:
            print("[Warning] Could not retrieve object positions:", e)
            positions = {}

        # === EXPLORATION PHASE: Try available skills and observe feedback ===
        print("\n[Exploration Phase] Begin skill probing to detect missing predicate...")

        # Skills available from provided list
        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper'
        ]

        # We will attempt each skill at least once with default/available objects or positions
        # If any action raises or returns a message about a missing-fluent or predicate, capture it.
        # This is a stub exploration: In realistic settings, the next lines would investigate all meaningful combinations.

        # We will scan for objects, drawers, locations from the observations (if possible)
        object_names = []
        drawer_names = []
        handle_names = []
        location_names = []
        if hasattr(task, "all_objects") and hasattr(task, "all_drawers"):
            try:
                object_names = task.all_objects
                drawer_names = task.all_drawers
                handle_names = getattr(task, "all_handles", [])
                location_names = getattr(task, "all_locations", [])
            except Exception as e:
                print("[Warning] Could not enumerate environment objects/drawers:", e)
        elif positions:
            # Fallback: Use keys from positions
            for name, pos in positions.items():
                lower_name = name.lower()
                if "drawer" in lower_name:
                    drawer_names.append(name)
                elif "handle" in lower_name:
                    handle_names.append(name)
                elif "location" in lower_name or "table" in lower_name or "pose" in lower_name:
                    location_names.append(name)
                else:
                    object_names.append(name)
        # Fallbacks
        if not location_names and positions:
            location_names = list(positions.keys())

        # Simple default args for exploration (these should match observation/provided env):
        o = object_names[0] if object_names else None
        d = drawer_names[0] if drawer_names else None
        h = handle_names[0] if handle_names else None
        p = location_names[0] if location_names else None
        p2 = location_names[1] if len(location_names) > 1 else p

        found_missing_predicate = False
        missing_predicate_name = None

        # Try each skill and see if any leads to a complaint about unknown predicates
        for skill in available_skills:
            try:
                print(f"[Exploration] Trying skill: {skill}")
                # Map to minimal legal args (since these are only for testing missing predicates)
                if skill == "execute_pick":
                    if o and p:
                        obs, reward, done = execute_pick(env, task, o, p)
                elif skill == "execute_place":
                    if o and d and p:
                        obs, reward, done = execute_place(env, task, o, d, p)
                elif skill == "execute_push":
                    if d and p:
                        obs, reward, done = execute_push(env, task, d, p)
                elif skill == "execute_pull":
                    # Try handle as object
                    if d and h and p:
                        obs, reward, done = execute_pull(env, task, d, h, p)
                elif skill == "execute_sweep":
                    if o and p:
                        obs, reward, done = execute_sweep(env, task, o, p)
                elif skill == "execute_rotate":
                    # If rotate needs something specific, skip for now
                    continue
                elif skill == "execute_go":
                    if p and p2 and p != p2:
                        obs, reward, done = execute_go(env, task, p, p2)
                elif skill == "execute_gripper":
                    obs, reward, done = execute_gripper(env, task)
                # Inspect results or exceptions for hints about missing predicates
            except Exception as ex:
                ex_msg = str(ex)
                print(f"[Exploration][{skill}] Exception: {ex_msg}")
                for predicate in ["identified", "temperature-known", "weight-known", "durability-known", "lock-known", "on-floor", "drawer-locked", "drawer-unlocked", "drawer-open", "drawer-closed"]:
                    if predicate in ex_msg:
                        found_missing_predicate = True
                        missing_predicate_name = predicate
                        print(f"[Exploration] Missing predicate detected: {predicate}")
                if "predicate" in ex_msg or "fluent" in ex_msg or "missing" in ex_msg:
                    found_missing_predicate = True
                    print(f"[Exploration] General predicate-related error detected: {ex_msg}")
                if found_missing_predicate:
                    break  # stop on first finding

        # === (After Exploration) Continue with Actual Task Plan ===
        if found_missing_predicate:
            print(f"[Task] Exploration detected missing predicate: {missing_predicate_name}. Please update your PDDL domain or provide this info.")
        else:
            print("[Task] No obvious missing predicate detected. Proceeding with oracle plan (if available).")

        # === MAIN TASK LOGIC: (Replace this with actual plan if available) ===
        # Here, we’re expected to execute a plan -- stub for demonstration:
        # For example, try a basic manipulation sequence as a fallback

        # For demonstration, let’s try a generic pick and place (assuming the environment has one pickable object and one drawer)
        try:
            if o and d and p and not found_missing_predicate:
                print(f"[Task] Picking up object '{o}' at location '{p}'.")
                obs, reward, done = execute_pick(env, task, o, p)

                print(f"[Task] Placing object '{o}' into drawer '{d}' at location '{p}'.")
                obs, reward, done = execute_place(env, task, o, d, p)

                if done:
                    print("[Task] Task succeeded after pick and place!")
                    return
                else:
                    print("[Task] Task did not finish after pick and place.")
            else:
                if not (o and d and p):
                    print("[Task] Not enough objects/locations available for main plan.")
        except Exception as ex:
            print("[Task] Exception while executing main plan:", ex)

        # --- End of main task logic ---

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()