# run_skeleton_task.py (Filled for Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined primitive skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === EXPLORATION PHASE ===
        # The feedback indicated exploration is necessary to find missing predicates
        # We'll use available skills (as described by the domain and exploration knowledge)
        # Only the predefined skills may be called

        # Assume all locations and objects are named in positions.keys()
        try:
            # Identify lists of objects and locations from positions
            # Expect positions: {'obj1': (pos1), 'obj2': (pos2), ...}
            object_names = []
            location_names = set()
            for key, value in positions.items():
                # Assume naming conventions: objects do not end with '_loc'
                if isinstance(value, tuple) or isinstance(value, list):
                    object_names.append(key)
                elif isinstance(value, str):
                    location_names.add(value)
            # For this code, let's assume keys are objects and we know some locations
            # Let's also mock the robot's starting location if possible
            # We'll need to know at least one location
            if location_names:
                start_location = list(location_names)[0]
            else:
                start_location = "ready-pose"  # fallback

        except Exception as e:
            print(f"[Exploration] Error parsing positions: {e}")
            object_names = []
            start_location = "ready-pose"

        # Define a simple exploration sequence using available skills
        # Since only predefined skills are allowed, we only use those:
        # Available: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 
        #            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We'll "visit" each object's location, and "pick" or "pull" each one in turn
        # We attempt to call each skill and gracefully handle failures to reveal missing predicates

        exploration_report = []

        for obj in object_names:
            # Try to get the position for the object and infer a reasonable "location" (for movement)
            try:
                obj_pos = positions[obj]
                # For locations, use the obj's location if present in positions dictionary;
                # otherwise, default to start_location.
                target_location = start_location

                # 1. Attempt to move to the object's location using execute_go
                # This may fail if the location is not valid, but that's part of exploration
                print(f"[Exploration] Attempting to move to object {obj} location.")
                try:
                    obs, reward, done, info = execute_go(
                        env,
                        task,
                        from_location=start_location,
                        to_location=target_location
                    )
                    print(f"[Exploration] Successfully moved to location of {obj}.")
                    exploration_report.append(f"Moved to {target_location} for {obj}")
                except Exception as move_exc:
                    print(f"[Exploration] Could not move to {target_location} for {obj}: {move_exc}")
                    exploration_report.append(f"FAILED: move to {target_location} for {obj}")

                # 2. Attempt to pick the object using execute_pick (predefined skill)
                print(f"[Exploration] Attempting to pick object {obj}.")
                try:
                    obs, reward, done, info = execute_pick(
                        env,
                        task,
                        object_name=obj,
                        location_name=target_location
                    )
                    print(f"[Exploration] Successfully picked {obj}.")
                    exploration_report.append(f"Picked {obj} at {target_location}")
                except Exception as pick_exc:
                    print(f"[Exploration] Could not pick {obj}: {pick_exc}")
                    exploration_report.append(f"FAILED: pick {obj} at {target_location}")

                # 3. Attempt to pull the object using execute_pull
                # This would test whether any relevant "lock-known" or related predicates are missing
                print(f"[Exploration] Attempting to pull object {obj}.")
                try:
                    obs, reward, done, info = execute_pull(
                        env,
                        task,
                        object_name=obj,
                        location_name=target_location
                    )
                    print(f"[Exploration] Successfully pulled {obj}.")
                    exploration_report.append(f"Pulled {obj} at {target_location}")
                except Exception as pull_exc:
                    print(f"[Exploration] Could not pull {obj}: {pull_exc}")
                    exploration_report.append(f"FAILED: pull {obj} at {target_location}")

                # Optionally, test other skills (execute_push, execute_sweep, etc.)
                print(f"[Exploration] Attempting to push at location {target_location}.")
                try:
                    obs, reward, done, info = execute_push(
                        env,
                        task,
                        location_name=target_location
                    )
                    print(f"[Exploration] Successfully pushed at {target_location}.")
                    exploration_report.append(f"Pushed at {target_location} after acting on {obj}")
                except Exception as push_exc:
                    print(f"[Exploration] Could not push at {target_location}: {push_exc}")
                    exploration_report.append(f"FAILED: push at {target_location} after {obj}")

            except Exception as e:
                print(f"[Exploration] Error in action sequence for {obj}: {e}")
                exploration_report.append(f"FAILED: general error with {obj}: {e}")

        # After exploring, print the report so the user can reason about missing predicates
        print("\n======= Exploration Report =======")
        for line in exploration_report:
            print(line)
        print("======= End Exploration Report =======")

        # If desired, the exploration can be repeated or logged for diagnostic purposes

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()