# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use all provided skills directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Optional video writers
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Get positions of objects and locations
        positions = get_object_positions()

        # === Begin Exploration Phase ===
        print("[Exploration] Checking for missing predicates by probing available actions...")

        # Attempt to use skill primitives and observe effects to illuminate missing predicates
        # Suppose we try available skills methodically and catch errors for missing predicates

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push',
            'execute_pull', 'execute_sweep', 'execute_rotate',
            'execute_go', 'execute_gripper'
        ]
        missing_predicate_found = False
        error_log = []

        # Try each skill with dummy (but plausible) arguments, catch exceptions, analyze errors
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                # For demo, pick plausible args for each action
                # We need to map these to what's in observation/positions
                # We'll use the keys/values in positions for objects and locations.
                # Names: objects, drawers, handles, locations; use first available for demo.
                objects = [k for k, v in positions.items() if 'obj' in k or 'ball' in k or 'object' in k]
                drawers = [k for k, v in positions.items() if 'drawer' in k]
                handles = [k for k, v in positions.items() if 'handle' in k]
                locations = [k for k, v in positions.items() if 'loc' in k or 'room' in k or 'pose' in k]
                # Fallback to all keys if not found
                if not objects: objects = list(positions.keys())
                if not drawers: drawers = list(positions.keys())
                if not locations: locations = list(positions.keys())
                if not handles: handles = list(positions.keys())

                # Call each skill with plausible arguments
                if skill == 'execute_pick':
                    # Pick an object at a location (simulate only)
                    obs, reward, done = execute_pick(
                        env, task,
                        target_obj=objects[0],
                        target_pos=positions[objects[0]],
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                elif skill == 'execute_place':
                    # Place object in drawer at a location
                    obs, reward, done = execute_place(
                        env, task,
                        held_obj=objects[0],
                        drawer=drawers[0],
                        target_pos=positions[drawers[0]],
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                elif skill == 'execute_push':
                    # Push a drawer at a location
                    obs, reward, done = execute_push(
                        env, task,
                        drawer=drawers[0],
                        push_pos=positions[drawers[0]],
                        push_distance=0.10,
                        timeout=10.0
                    )
                elif skill == 'execute_pull':
                    # Pull a handle or drawer at location
                    obs, reward, done = execute_pull(
                        env, task,
                        handle=handles[0],
                        drawer=drawers[0],
                        pull_pos=positions[handles[0]],
                        pull_distance=0.10,
                        timeout=10.0
                    )
                elif skill == 'execute_sweep':
                    # Sweep an object at a location
                    obs, reward, done = execute_sweep(
                        env, task,
                        target_obj=objects[0],
                        target_pos=positions[objects[0]],
                        sweep_distance=0.12,
                        timeout=10.0
                    )
                elif skill == 'execute_rotate':
                    # Rotate an object at a location
                    obs, reward, done = execute_rotate(
                        env, task,
                        rot_obj=objects[0],
                        target_axis='z',
                        angle=np.pi/2,
                        timeout=10.0
                    )
                elif skill == 'execute_go':
                    # Move from one location to another
                    obs, reward, done = execute_go(
                        env, task,
                        from_pos=positions[locations[0]],
                        to_pos=positions[locations[-1]],
                        timeout=10.0
                    )
                elif skill == 'execute_gripper':
                    # Use gripper action (no args usually)
                    obs, reward, done = execute_gripper(
                        env, task,
                        timeout=5.0
                    )
                print(f"[Exploration] Skill {skill} succeeded.")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed with error: {e}")
                error_log.append((skill, str(e)))
                # Example: If error suggests missing predicate/place/attribute, capture
                if "predicate" in str(e) or "precondition" in str(e) or "unknown" in str(e):
                    missing_predicate_found = True

        # Print findings of exploration
        print("[Exploration] Exploration phase completed.")
        if missing_predicate_found:
            print("[Exploration] Potential missing predicate or state discovered. Review error log for details.")
        else:
            print("[Exploration] No explicit missing predicate encountered using provided skills.")

        print("[Exploration] Error log:")
        for skill, err in error_log:
            print(f"  Skill: {skill} | Error: {err}")

        # Optionally: You could automate forming a report or use error messages to inform next planning step.

        # === Task Plan Execution ===
        # If you have the oracle plan/steps, execute them in order using the corresponding skills. For now, this
        # is a placeholder: you would need a plan array/structure.
        # E.g.:
        # plan = [
        #     ("execute_go", args...),
        #     ("execute_pick", args...),
        #     ...
        # ]
        # for action_name, *params in plan:
        #     try:
        #         print(f"[Task] Executing: {action_name} with params {params}")
        #         fn = globals()[action_name]
        #         obs, reward, done = fn(env, task, *params)
        #         if done:
        #             print("[Task] Task complete after:", action_name)
        #             break
        #     except Exception as e:
        #         print(f"[Task] Error executing action {action_name}: {e}")
        #         break

        # TODO: Insert actual plan and step through with skill calls as above.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()