# run_skeleton_task.py (Completed Task with Exploration for (need-ready) Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *   # Use only predefined actions/skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including an exploration phase to identify missing predicates (e.g., need-ready).'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dictionary mapping object names to their positions or info
        positions = get_object_positions()

        # === Exploration Phase ===
        #
        # The feedback indicates that the (need-ready) predicate is missing, which acts as a precondition-blocker for some actions. 
        # The exploration plan will apply actions (from the available skill set) that could reveal or trigger the predicate in the domain.
        # We will attempt actions and check the state transitions or errors to infer when (need-ready) becomes set or cleared.
        #
        # This phase simulates what would happen when a predicate like (need-ready) blocks actions, e.g. after pick/place.

        print("[Exploration] Begin exploration to detect the missing (need-ready) predicate trigger circumstances.")

        # Define the set of objects, drawers, handles, and locations using the positions dict
        objects = [name for name in positions if positions[name].get('type', '') == 'object' and not positions[name].get('is_handle', False)]
        handles = [name for name in positions if positions[name].get('type', '') == 'object' and positions[name].get('is_handle', False)]
        drawers = [name for name in positions if positions[name].get('type', '') == 'drawer']
        locations = list(set([positions[name]['location'] for name in positions if 'location' in positions[name]]))
        ready_pose = 'ready-pose'
        robot_initial_location = None
        for name in positions:
            if positions[name].get('robot_start', False):
                robot_initial_location = positions[name]['location']
                break
        # Fallback if not found
        if robot_initial_location is None:
            robot_initial_location = locations[0] if len(locations) > 0 else None

        # 1. Try picking up each object and see if the system blocks further actions
        for obj in objects:
            try:
                print(f"[Exploration] Attempting execute_pick on object {obj} at location {robot_initial_location}...")
                obs, reward, done, info = execute_pick(env, task, obj, robot_initial_location)
                print(f"[Exploration] execute_pick completed on {obj}. Now, test if another pick is possible without ready action.")
                # Try to pick another object immediately (should be blocked by (need-ready) if the domain enforces it)
                for obj2 in objects:
                    if obj2 != obj:
                        try:
                            obs2, reward2, done2, info2 = execute_pick(env, task, obj2, robot_initial_location)
                            print(f"[Exploration] Unexpected: execute_pick did NOT fail on {obj2} after picking {obj}. (need-ready may not be enforced)")
                        except Exception as e2:
                            print(f"[Exploration] As expected, execute_pick on {obj2} was blocked after picking {obj}: {e2}")
                # Now try to perform execute_go (move)
                try:
                    print(f"[Exploration] Try execute_go to another location ({ready_pose}) after pick...")
                    obs3, reward3, done3, info3 = execute_go(env, task, robot_initial_location, ready_pose)
                    print("[Exploration] execute_go succeeded (need-ready NOT blocking move) -- unexpected if (need-ready) enforced.")
                except Exception as e3:
                    print(f"[Exploration] execute_go failed after pick as expected (blocked by (need-ready)): {e3}")
                # Now, perform the ready-pose action and try again
                try:
                    print("[Exploration] Trying execute_go_ready to clear (need-ready)...")
                    obs4, reward4, done4, info4 = execute_go_ready(env, task, robot_initial_location)
                    print("[Exploration] execute_go_ready succeeded. (need-ready) likely cleared.")
                except Exception as e4:
                    print(f"[Exploration] execute_go_ready failed: {e4}")
                # Attempt another pick/go now
                for obj3 in objects:
                    if obj3 != obj:
                        try:
                            obs5, reward5, done5, info5 = execute_pick(env, task, obj3, robot_initial_location)
                            print(f"[Exploration] Post-ready, execute_pick on {obj3} succeeded as expected.")
                        except Exception as e5:
                            print(f"[Exploration] Post-ready, execute_pick on {obj3} failed unexpectedly: {e5}")
                try:
                    obs6, reward6, done6, info6 = execute_go(env, task, robot_initial_location, locations[0])
                    print(f"[Exploration] Post-ready, execute_go to {locations[0]} succeeded as expected.")
                except Exception as e6:
                    print(f"[Exploration] Post-ready, execute_go failed unexpectedly: {e6}")
            except Exception as e:
                print(f"[Exploration] execute_pick on {obj} failed: {e}")

        # 2. Try place, push, pull, or other relevant skills and see when (need-ready) is set/cleared.
        for obj in objects:
            for drawer in drawers:
                try:
                    print(f"[Exploration] Attempting execute_place of {obj} into drawer {drawer} at location {robot_initial_location}...")
                    obs7, reward7, done7, info7 = execute_place(env, task, obj, drawer, robot_initial_location)
                    print(f"[Exploration] execute_place completed for {obj} into {drawer}. Next, check if (need-ready) blocks new actions.")
                    # Try another place without ready
                    for obj4 in objects:
                        if obj4 != obj:
                            try:
                                obs8, reward8, done8, info8 = execute_place(env, task, obj4, drawer, robot_initial_location)
                                print("[Exploration] Unexpected: execute_place did NOT fail (need-ready may not be enforced).")
                            except Exception as e8:
                                print(f"[Exploration] As expected, execute_place blocked for {obj4}: {e8}")
                    try:
                        print(f"[Exploration] Trying execute_go_ready after place to clear (need-ready)...")
                        obs9, reward9, done9, info9 = execute_go_ready(env, task, robot_initial_location)
                        print("[Exploration] execute_go_ready after place succeeded.")
                    except Exception as e9:
                        print(f"[Exploration] execute_go_ready after place failed: {e9}")
                except Exception as e7:
                    print(f"[Exploration] execute_place on {obj} failed: {e7}")

        # 3. Try handle and drawer actions if present
        for handle in handles:
            for drawer in drawers:
                try:
                    print(f"[Exploration] Trying execute_pull on {drawer} using handle {handle} at {robot_initial_location}...")
                    obs10, reward10, done10, info10 = execute_pull(env, task, drawer, handle, robot_initial_location)
                    print("[Exploration] execute_pull completed. Testing if (need-ready) blocks further drawer actions.")
                    # Try another pull immediately
                    try:
                        obs11, reward11, done11, info11 = execute_pull(env, task, drawer, handle, robot_initial_location)
                        print(f"[Exploration] Unexpected: execute_pull did not fail.")
                    except Exception as e11:
                        print(f"[Exploration] As expected, execute_pull blocked after previous: {e11}")
                except Exception as e10:
                    print(f"[Exploration] execute_pull failed: {e10}")

        print("[Exploration] End of predicate exploration for (need-ready).\n")
        print("You should now have sufficient knowledge about the missing predicate's (need-ready) conditions in the domain and can design your task plans accordingly.\n")

        # === NEXT: Continue normal oracle/task plan execution as needed (omitted per instructions) ===

        # TODO: Continue your oracle/task plan here as required.
        # Example:
        # obs, reward, done, info = execute_pick(env, task, some_object, some_location)
        # if done:
        #     print("[Task] Task complete after exploration.")
        #     return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
