# run_skeleton_task.py (Completed for Exploration Phase - Find Missing Predicate)
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Do not redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ------------------------------------------------------------
        # === Exploration Phase: Find the Missing Predicate        ===
        # ------------------------------------------------------------
        # Based on the feedback: (drawer-open drawer_bottom)
        # The goal is to explore and try to find how to achieve the literal (drawer-open drawer_bottom)
        # This phase will try all plausible actions, check their effects, and look for the missing predicate.

        # We'll attempt to do this in a way that is robust to errors, and logs any predicate changes.

        # To keep things generic and robust, let's do the following:
        # - Try skill actions that could potentially change the drawer state.
        # - Observe if (drawer-open drawer_bottom) holds after any action.
        # - Record and print predicates that become true.

        # We'll use positions info to attempt action skills.
        # Because we don't have detailed object names, we use plausible names and handle exceptions.
        # Expect the environment to have: a drawer (likely named "drawer_bottom"), one or more handles, and robot at some initial pose.

        # ------------ Discover objects and initial state ------------
        drawer_name = None
        handle_name = None
        robot_location = None

        # Try to guess likely object names
        for obj in positions:
            lname = obj.lower()
            if 'drawer' in lname and (drawer_name is None):  # Prefer the first drawer
                drawer_name = obj
            if 'handle' in lname and (handle_name is None):
                handle_name = obj
        # Fallbacks in case names aren't canonical
        if drawer_name is None:
            drawer_name = 'drawer_bottom'
        if handle_name is None:
            handle_name = 'handle_bottom'

        # Try to determine the robot's initial location
        try:
            robot_location = task.get_robot_location() if hasattr(task, 'get_robot_location') else 'ready-pose'
        except Exception:
            robot_location = 'ready-pose'

        # Try to determine the drawer location (for action specificity)
        if drawer_name in positions:
            drawer_pos = positions[drawer_name]
        else:
            drawer_pos = None

        # ---------- Exploration: Try skills and look for state change ----------
        exploration_steps = [
            # Move to the drawer (if needed)
            ('execute_go', [robot_location, drawer_name]),
            # Try to pull the drawer (unlocking/opening)
            ('execute_pull', [drawer_name, handle_name, drawer_name]),
            # Try to push the drawer (if already open/close)
            ('execute_push', [drawer_name, drawer_name]),
            # Try to pick the handle (for pulling)
            ('execute_pick', [handle_name, drawer_name]),
            # Try to pick the drawer directly (may fail)
            ('execute_pick', [drawer_name, drawer_name]),
            # Try to place something (if holding)
            ('execute_place', [drawer_name, drawer_name, drawer_name]),
            # Try to sweep (edge case)
            ('execute_sweep', [drawer_name, drawer_name]),
            # Try to use gripper primitive
            ('execute_gripper', []),
        ]
        # Only use predefined skills. Each is executed with best-guess parameters.

        found_predicate = False

        for skill, args in exploration_steps:
            try:
                print(f"[Exploration] Attempting skill: {skill} with args: {args}")
                # Call the skill if it exists in skill_code
                skill_fn = globals().get(skill)
                if skill_fn is None:
                    print(f"  [WARN] Skill {skill} not found in available skills. Skipping.")
                    continue

                result = None
                if skill == 'execute_go':
                    # Go (move) expects from and to (location names)
                    result = skill_fn(env, task, *args)
                elif skill in ['execute_pick', 'execute_place', 'execute_pull', 'execute_push', 'execute_sweep']:
                    # Generic skill: pass env, task, plus required args
                    result = skill_fn(env, task, *args)
                elif skill == 'execute_gripper':
                    result = skill_fn(env, task)
                else:
                    # Other skills not handled
                    continue

                # After execution, update robot's current location if moved
                if skill == 'execute_go':
                    robot_location = args[1]

                # After any action, get the observation and check for the missing predicate
                curr_obs = task.get_observation()
                # The observation may have a property or a set of predicates
                # Try to find (drawer-open drawer_bottom) or equivalent
                obs_predicates = []
                if hasattr(curr_obs, "predicates"):
                    obs_predicates = curr_obs.predicates
                elif isinstance(curr_obs, dict) and "predicates" in curr_obs:
                    obs_predicates = curr_obs["predicates"]
                else:
                    obs_predicates = str(curr_obs)

                # Feedback target: looking for drawer-open
                predicate_str = f"(drawer-open {drawer_name})"
                if predicate_str in str(obs_predicates):
                    print(f"\n[RESULT] Found missing predicate: {predicate_str}\n")
                    found_predicate = True
                    break
                elif "(drawer-open drawer_bottom)" in str(obs_predicates):
                    print(f"\n[RESULT] Found missing predicate: (drawer-open drawer_bottom)\n")
                    found_predicate = True
                    break
                else:
                    print(f"[Observation] Current predicates do not yet show drawer-open for {drawer_name}.")
            except Exception as ex:
                print(f"[ERROR] Exception while executing skill {skill}: {ex}")

        if not found_predicate:
            print("\n[FAIL] Could not discover the (drawer-open drawer_bottom) predicate using available skills.\nConsider reviewing skill parameters or initial state.")

        # ------------------------------------------------------------
        # === End of Exploration Phase                              ===
        # ------------------------------------------------------------

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
