# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use existing skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions is a dict: object names -> positions

        # === Exploration Phase for Missing Predicate ===

        print("[Exploration] Starting predicate exploration...")

        # Assumptions: executing available skills per action list,
        # with logic inferred from exploration knowledge domain.
        # The missing predicate may be e.g., 'lock-known' for drawers.
        # We attempt to systematically execute skills to discover which step is blocked/missing.

        # Example environment object analysis
        # Find drawers, handles, objects on floor, and possible locations
        drawers = []
        handles = []
        objects_on_floor = []
        locations = set()
        for obj_name in positions.keys():
            if 'drawer' in obj_name:
                drawers.append(obj_name)
            if 'handle' in obj_name:
                handles.append(obj_name)
            if 'obj' in obj_name:
                objects_on_floor.append(obj_name)
            # Assume all object positions are at "locations"
            loc = positions[obj_name]
            locations.add(loc)
        locations = list(locations)

        # Try to use each available skill, observe success/failure
        exploration_results = {}
        # For demonstration, pick one handle and drawer to try
        drawer = drawers[0] if drawers else None
        handle = handles[0] if handles else None
        obj = objects_on_floor[0] if objects_on_floor else None
        loc = None
        if drawer:
            loc = positions[drawer]
        elif obj:
            loc = positions[obj]
        elif handle:
            loc = positions[handle]
        else:
            loc = None

        # Keep track of exceptions to deduce missing predicates
        missing_predicate = None

        # Try step-by-step exploration
        try:
            # 1. Try execute_pick on object (if available)
            if obj and hasattr(env, 'hand_empty') and env.hand_empty:
                print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=obj,
                    location=loc
                )
                exploration_results['execute_pick'] = (obs, reward, done)
        except Exception as e:
            print(f"[Exploration] execute_pick failed with exception: {e}")
            missing_predicate = "handempty or related precondition"

        try:
            # 2. Try execute_pick on handle (for drawer)
            if handle:
                print(f"[Exploration] Trying execute_pick on handle {handle} at {positions[handle]}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=handle,
                    location=positions[handle]
                )
                exploration_results['execute_pick_handle'] = (obs, reward, done)
        except Exception as e:
            print(f"[Exploration] execute_pick_handle failed with exception: {e}")
            missing_predicate = "is-handle or related precondition"

        try:
            # 3. Try execute_pull (to open drawer using handle)
            if drawer and handle:
                print(f"[Exploration] Trying execute_pull on drawer {drawer}, handle {handle} at {positions[drawer]}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer,
                    handle_name=handle,
                    location=positions[drawer]
                )
                exploration_results['execute_pull'] = (obs, reward, done)
        except Exception as e:
            print(f"[Exploration] execute_pull failed with exception: {e}")
            missing_predicate = "drawer-unlocked or drawer-closed or lock-known"

        try:
            # 4. Try execute_push (to close the drawer)
            if drawer:
                print(f"[Exploration] Trying execute_push on drawer {drawer} at {positions[drawer]}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawer,
                    location=positions[drawer]
                )
                exploration_results['execute_push'] = (obs, reward, done)
        except Exception as e:
            print(f"[Exploration] execute_push failed with exception: {e}")
            missing_predicate = "drawer-open or robot-free or need-ready"

        try:
            # 5. Try execute_place (place an object into drawer)
            if obj and drawer:
                print(f"[Exploration] Trying execute_place with {obj} into {drawer} at {positions[drawer]}")
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=obj,
                    drawer_name=drawer,
                    location=positions[drawer]
                )
                exploration_results['execute_place'] = (obs, reward, done)
        except Exception as e:
            print(f"[Exploration] execute_place failed with exception: {e}")
            missing_predicate = "holding, drawer-open, not drawer-full, robot-at, not need-ready"

        try:
            # 6. Try move (execute_go): move robot from one location to another
            if len(locations) >= 2:
                loc_from, loc_to = locations[0], locations[1]
                print(f"[Exploration] Trying execute_go from {loc_from} to {loc_to}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=loc_from,
                    to_location=loc_to
                )
                exploration_results['execute_go'] = (obs, reward, done)
        except Exception as e:
            print(f"[Exploration] execute_go failed with exception: {e}")
            missing_predicate = "robot-free or need-ready or robot-at"

        # Report on exploration conclusions
        if missing_predicate:
            print(f"[Exploration] Potential missing predicate identified: {missing_predicate
}")
        else:
            print("[Exploration] All skills executed without precondition failures; no missing predicates evident.")

        # Optionally show all exploration results
        print("[Exploration] Exploration results summary:")
        for skill, result in exploration_results.items():
            print(f"    {skill}: {result}")

        # Continue with main oracle plan execution if desired...
        # (Not implemented: just exploration in this phase)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
