# run_skeleton_task.py (Filled: Exploration for (robot-at ready-pose) predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only skills provided by this module

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # print("[Exploration] Object positions in environment:", positions)
        #
        # Plan:
        # We need to determine the missing predicate - feedback indicates (robot-at ready-pose) may be problematic.
        # The 'execute_go_ready' action can set the 'robot-at ready-pose' predicate.
        # Exploration here means: Try to run 'execute_go_ready' and confirm.
        #
        # Only call skills defined in skill_code:
        # - execute_go
        # - execute_go_ready
        # etc.

        # Get the robot's current location (if possible)
        robot_location = None
        for k in positions:
            if k.startswith('robot') or 'robot' in k:
                robot_info = positions[k]
                if isinstance(robot_info, dict) and 'location' in robot_info:
                    robot_location = robot_info['location']
                else:
                    # Otherwise, try to infer from key or value
                    robot_location = k
                break

        # If not, check observation for current robot location
        if robot_location is None and isinstance(obs, dict):
            if 'robot_at' in obs:
                robot_location = obs['robot_at']
            elif 'robot-at' in obs:
                robot_location = obs['robot-at']

        # For exploration, we assume start location if none found
        if robot_location is None:
            robot_location = 'init_pose'
        
        # Step 1: Move to some other location (if not already at ready-pose)
        # - We want to trigger 'robot-at ready-pose'
        try:
            # First, check if already at ready-pose -- if yes, skip; else move.
            if robot_location != 'ready-pose':
                print("[Exploration] Moving robot to ready-pose")
                # The primitive is execute_go_ready(current_location)
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=robot_location,
                    timeout=10.0
                )
                print("[Exploration] Called execute_go_ready(from=%s)" % robot_location)
            else:
                print("[Exploration] Robot already at ready-pose.")
        except Exception as e:
            print("[Exploration] Exception running execute_go_ready:", e)

        # Check if the predicate (robot-at ready-pose) is now true
        # If object_positions or the obs dict provide this info, print it.
        ready_pose_state = False
        if 'robot-at' in obs and obs['robot-at'] == 'ready-pose':
            ready_pose_state = True
        elif 'robot_at' in obs and obs['robot_at'] == 'ready-pose':
            ready_pose_state = True
        elif hasattr(task, 'get_current_location'):
            try:
                loc = task.get_current_location()
                if loc == 'ready-pose':
                    ready_pose_state = True
            except Exception:
                pass

        print("[Exploration] After execute_go_ready: (robot-at ready-pose) =", ready_pose_state)

        if not ready_pose_state:
            print("[ERROR] (robot-at ready-pose) predicate not true after execute_go_ready. There may be a domain or observation bug.")

        # If there are other missing predicates (e.g., for skill preconditions in apply plan),
        # similar exploration can be done using available skills and checking system feedback.

        # The skeleton can be extended by running other available skills for further exploration,
        # depending on what feedback or failures are encountered.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
