# run_skeleton_task.py (Executable Exploration with Predicate Discovery)
import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for predicate discovery.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        print("[Exploration] Starting predicate exploration...")

        # -------------------------------------------------------
        # Exploration phase to discover missing predicates
        # -------------------------------------------------------
        #
        # Based on exploration knowledge and feedback:
        # The missing predicate is likely (robot-at ready-pose)
        # or a location called 'ready-pose'. We'll explore locations
        # and try to invoke the skills to check if "ready-pose" exists.

        found_ready_pose = False
        ready_pose_key = None

        # Try to find a location named 'ready-pose' in the known positions
        for key in positions.keys():
            if "ready" in key.lower() and "pose" in key.lower():
                ready_pose_key = key
                found_ready_pose = True
                print(f"[Exploration] Found candidate ready-pose location: {ready_pose_key}")
                break

        # If not found by name, may need to try all locations for the relevant skills
        if not found_ready_pose:
            for key in positions.keys():
                if "pose" in key.lower() or "home" in key.lower():
                    ready_pose_key = key
                    found_ready_pose = True
                    print(f"[Exploration] Found fallback ready-pose location: {ready_pose_key}")
                    break

        if found_ready_pose:
            print(f"[Exploration] Testing move to ready-pose: {ready_pose_key}")
            try:
                # Try to go to ready-pose using 'execute_go' or 'execute_go_ready'
                # Execute only if the function exists in skill_code
                # Try execute_go (from current to ready_pose)
                robot_current_location = None
                for key in positions.keys():
                    if "robot" in key.lower() and "at" in key.lower():
                        robot_current_location = positions[key]
                        break
                # Try to move to ready-pose
                obs, reward, done = execute_go(env, task, from_location=None, to_location=ready_pose_key)
                print("[Exploration] execute_go to ready-pose success.")
            except Exception as e:
                print(f"[Error] Moving to ready-pose using execute_go failed: {e}")
            try:
                obs, reward, done = execute_go_ready(env, task, from_location=ready_pose_key)
                print("[Exploration] execute_go_ready to ready-pose success.")
            except Exception as e:
                print(f"[Error] Calling execute_go_ready failed (possibly not implemented): {e}")

        else:
            print("[Exploration] No ready-pose found by name in object_positions.")

        print("[Exploration] If task requires visiting 'ready-pose', the predicate (robot-at ready-pose) or variable (ready-pose - location) is likely missing. Please review domain/observation.")
        
        # ---------------
        # (After exploring) - Example plan skeleton using provided skills
        # ---------------
        # Once the missing predicate is identified, the task plan proceeds as normal,
        # moving the robot with skills like execute_go/execute_go_ready when required.

        # Place your oracle plan here, using ONLY provided skills, e.g.:
        # For demonstration, suppose the plan requires:
        # - Picking an object
        # - Placing it in a drawer
        # - Going to the ready-pose at end

        # 1. Find object and drawer from positions (example names)
        obj_name = None
        drawer_name = None
        obj_location = None
        drawer_location = None
        for k in positions.keys():
            if "obj" in k.lower() or "ball" in k.lower():
                obj_name = k
                obj_location = k  # For this env structure, keys serve as semantic locations
                break
        for k in positions.keys():
            if "drawer" in k.lower():
                drawer_name = k
                drawer_location = k
                break

        # 2. Example workflow with error handling
        try:
            if obj_name and obj_location:
                print(f"[Task] Picking object: {obj_name} at {obj_location}")
                obs, reward, done = execute_pick(env, task, obj_name, obj_location)
            else:
                print("[Task] No pickable object found in positions.")

            if drawer_name and drawer_location and obj_name:
                print(f"[Task] Placing object: {obj_name} into drawer: {drawer_name} at {drawer_location}")
                obs, reward, done = execute_place(env, task, obj_name, drawer_name, drawer_location)
            else:
                print("[Task] No suitable drawer/object for placement found.")

            if ready_pose_key:
                print(f"[Task] Go to ready-pose: {ready_pose_key}")
                try:
                    obs, reward, done = execute_go_ready(env, task, from_location=drawer_location or obj_location)
                except Exception as e:
                    print(f"[Error] Could not call execute_go_ready: {e}")
            else:
                print("[Task] Skipping go to ready-pose step as none found or explored.")
        except Exception as task_e:
            print(f"[Task Error] During oracle plan execution: {task_e}")

        # ---------------
        # End of main exploration/task logic
        # ---------------

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
