# run_skeleton_task.py (Completed for Exploration Phase & Finding Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for predicate checking.'''
    print("===== Starting Skeleton Task (Exploration) =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try Predicates One by One ===

        # The purpose here is to determine which predicate is missing in the domain/problem definition
        # Based on exploration knowledge, we will try skills that can surface missing predicates.

        # Start by retrieving environment objects and locations
        # objects: Each object in the scene
        # locations: Each location/region ("drawer", "table", etc.)
        objects = positions.get('objects', [])
        locations = positions.get('locations', [])
        robot_name = positions.get('robot', 'robot')  # fallback robot name

        # We must try to perform actions that would fail (or yield errors)
        # if a certain predicate is missing. For example, checking if the robot "knows"
        # about some property after an action.

        # For instance, try to execute the following exploration skills in the environment:
        #   1. execute_go_identify  (identify if an object is at a location)
        #   2. execute_go_temperature (try to learn temperature-known)
        #   3. execute_pick_weight (try to pick and obtain weight-known)
        #   4. execute_pick_durability (try to pick and obtain durability-known)
        #   5. execute_pull (to reveal lock-known)

        # We'll try to execute each skill and observe whether the predicate for that skill is affected.
        # Since feedback mentions lock-known, and the exploration actions include one for lock-known,
        # We'll organize the exploration accordingly.

        failed_predicates = []

        # 1. Try to identify all objects at each location (execute_go_identify)
        for loc in locations:
            try:
                obs, reward, done, info = execute_go(env, task, from_loc=positions.get('initial_robot_location', loc),
                                                     to_loc=loc, timeout=10.0)
                print(f"[Exploration] Robot moved to {loc} to try to identify objects.")
                # Check if some objects at this location now report "identified"
                # Here we would check obs or info ("identified" field) -- if available
                # If not, we proceed to the next exploration
            except Exception as e:
                print(f"[Exploration] Failed to identify objects at {loc}: {e}")
                failed_predicates.append('identified')
        
        # 2. Try to pick objects to reveal weight-known and durability-known
        for obj in objects:
            obj_pos = positions.get(obj)
            loc = None
            # Try to find out at which location this object is
            for loc_name in locations:
                obj_list = positions.get('objects_at_' + loc_name, [])
                if obj in obj_list:
                    loc = loc_name
                    break
            if loc is None:
                continue
            # (a) Try to pick and get weight-known
            try:
                obs, reward, done, info = execute_pick(env, task, obj_name=obj, location=loc, timeout=10.0)
                print(f"[Exploration] Attempted to pick {obj} at {loc} for weight-known.")
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj} for weight-known: {e}")
                failed_predicates.append('weight-known')
            # (b) Try to pick and get durability-known (if separate)
            try:
                obs, reward, done, info = execute_pick(env, task, obj_name=obj, location=loc, durability_check=True, timeout=10.0)
                print(f"[Exploration] Attempted to pick {obj} at {loc} for durability-known.")
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj} for durability-known: {e}")
                failed_predicates.append('durability-known')
            
        # 3. For lock-known, try to pull objects (possibly handles)
        for obj in objects:
            loc = None
            for loc_name in locations:
                obj_list = positions.get('objects_at_' + loc_name, [])
                if obj in obj_list:
                    loc = loc_name
                    break
            if loc is None:
                continue
            try:
                # Try to pick the handle or pull the object (simulate testing the lock)
                obs, reward, done, info = execute_pull(env, task, obj_name=obj, location=loc, timeout=10.0)
                print(f"[Exploration] Attempted to pull {obj} at {loc} for lock-known.")
            except Exception as e:
                print(f"[Exploration] Failed to pull {obj} for lock-known: {e}")
                failed_predicates.append('lock-known')

        # 4. Try to observe temperature (execute_go_temperature)
        for loc in locations:
            try:
                obs, reward, done, info = execute_go(env, task, from_loc=positions.get('initial_robot_location', loc),
                                                     to_loc=loc, temperature_check=True, timeout=10.0)
                print(f"[Exploration] Robot moved to {loc} to check temperature-known.")
            except Exception as e:
                print(f"[Exploration] Failed to check temperature at {loc}: {e}")
                failed_predicates.append('temperature-known')

        # ---- Exploration results summary ----
        if failed_predicates:
            missing = set(failed_predicates)
            print("===== [Exploration Summary] Missing or problematic predicates detected: =====")
            for pred in missing:
                print(" -", pred)
            print("You should check domain/problem PDDL for missing/incorrect predicates among those listed above.")
        else:
            print("[Exploration] No missing predicates detected by skill execution.")

        print("===== [Exploration Complete] =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
