# run_skeleton_task.py (Completed for Exploration and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # You don't need to redefine primitives like move, pick, place

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase (Discovering Missing Predicate) ===
        # Based on the domain and exploration knowledge, we must determine if extra predicates are needed
        # and which ones might be missing by attempting available exploratory skills and monitoring their effects.

        # For demonstration, we explore by calling skills that might reveal missing predicates.
        # We'll iterate over the known objects and locations, applying the available skills.

        # Get object and location lists from positions info
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        handle_names = [name for name in positions if positions[name].get('type', '') == 'handle']

        # You might need to adjust the extraction of 'type' and names based on your positions dictionary structure.

        missing_predicates = []
        exploration_complete = False

        print("[Exploration] Starting exploration for missing predicates...")

        # Try to execute exploration with each available skill and log the result.
        # We assume 'env' and 'task' can be passed to skills (see earlier skill_code.py).

        # Example: try to pick up objects - if some error or block occurs, we assume predicate is missing.
        for obj in object_names:
            # Try picking with "execute_pick" first, by trying both an object and a location.
            try:
                for loc in location_names:
                    print(f"[Exploration] Trying execute_pick: object={obj}, location={loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    # If the pick succeeds, we check for unresolved properties
                    # If exception or done indicates some state is unreachable, note that as missing predicate
                    if done:
                        print(f"[Exploration] Pick executed with done=True for {obj}.")
                        exploration_complete = True
                        break
            except Exception as e:
                print(f"[Exploration] Exception in execute_pick for {obj} at {loc}: {e}")
                missing_predicates.append('execute_pick_precondition_failed')

        # Try to execute handle pick if handle exists
        for handle in handle_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_pick_handle: handle={handle}, location={loc}")
                    obs, reward, done = execute_pick_handle(env, task, handle, loc)
                    if done:
                        print(f"[Exploration] Handle pick executed with done=True for {handle}.")
                        exploration_complete = True
                        break
                except Exception as e:
                    print(f"[Exploration] Exception in execute_pick_handle for {handle} at {loc}: {e}")
                    missing_predicates.append('execute_pick_handle_precondition_failed')

        # Try pulling each drawer-handle pair (if both exist)
        for d in drawer_names:
            for h in handle_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_pull: drawer={d}, handle={h}, location={loc}")
                        obs, reward, done = execute_pull(env, task, d, h, loc)
                        if done:
                            print(f"[Exploration] Pull executed with done=True for drawer={d} handle={h} location={loc}.")
                            exploration_complete = True
                            break
                    except Exception as e:
                        print(f"[Exploration] Exception in execute_pull for drawer={d}, handle={h} at {loc}: {e}")
                        missing_predicates.append('execute_pull_precondition_failed')

        # Try pushing drawers
        for d in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_push: drawer={d}, location={loc}")
                    obs, reward, done = execute_push(env, task, d, loc)
                    if done:
                        print(f"[Exploration] Push executed with done=True for drawer={d} at {loc}.")
                        exploration_complete = True
                        break
                except Exception as e:
                    print(f"[Exploration] Exception in execute_push for drawer={d} at {loc}: {e}")
                    missing_predicates.append('execute_push_precondition_failed')

        # Try going from every location to every location
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc == to_loc:
                    continue
                try:
                    print(f"[Exploration] Trying execute_go: from={from_loc}, to={to_loc}")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    if done:
                        print(f"[Exploration] Go executed with done=True from {from_loc} to {to_loc}.")
                        exploration_complete = True
                        break
                except Exception as e:
                    print(f"[Exploration] Exception in execute_go from {from_loc} to {to_loc}: {e}")
                    missing_predicates.append('execute_go_precondition_failed')

        # Try gripper skill if available
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Gripper executed with done=True.")
        except Exception as e:
            print(f"[Exploration] Exception in execute_gripper: {e}")
            missing_predicates.append('execute_gripper_precondition_failed')

        # Try sweep skill if available
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_sweep: object={obj}, location={loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Sweep executed with done=True for object={obj} at location={loc}.")
                        exploration_complete = True
                        break
                except Exception as e:
                    print(f"[Exploration] Exception in execute_sweep for {obj} at {loc}: {e}")
                    missing_predicates.append('execute_sweep_precondition_failed')

        # Try rotate (if available in skills)
        if 'execute_rotate' in globals():
            for obj in object_names:
                try:
                    print(f"[Exploration] Trying execute_rotate: object={obj}")
                    obs, reward, done = execute_rotate(env, task, obj)
                    if done:
                        print(f"[Exploration] Rotate executed with done=True for object={obj}.")
                        exploration_complete = True
                        break
                except Exception as e:
                    print(f"[Exploration] Exception in execute_rotate for {obj}: {e}")
                    missing_predicates.append('execute_rotate_precondition_failed')

        # Log out any missing predicates identified
        print("[Exploration] Exploration finished. Summary of missing predicates or failed preconditions:")
        for mp in set(missing_predicates):
            print(f"  - {mp}")

        if not missing_predicates:
            print("[Exploration] No missing predicates detected; all known skills ran without precondition errors.")

        # === End of Exploration Phase ===
        # At this point you would synthesize new planning knowledge, e.g., update the domain or PDDL with the missing predicate.

        print("===== End of Skeleton Task =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)


if __name__ == "__main__":
    run_skeleton_task()
