# run_skeleton_task.py (Completed for Exploration to Discover Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Directly using available skills, do not redefine!

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Find Missing Predicate (drawer-unlocked ?) ---
        # According to feedback, we need to discover (drawer-unlocked drawer1).
        # We'll attempt to use predefined actions and skills to "explore" and determine if the drawer is unlocked.

        # For demonstration, we suppose the following names for locations and objects
        # (as real values would come from observation/positions)
        robot_location = 'ready-pose'
        drawer_name = 'drawer1'
        handle_name = None
        
        # Try to identify the handle object for drawer1
        for obj, pos in positions.items():
            if "handle" in obj and drawer_name in obj:
                handle_name = obj
                break
        if handle_name is None:
            # Fallback: just pick any object with 'handle' in the name
            for obj in positions:
                if "handle" in obj:
                    handle_name = obj
                    break
                
        # Also find a location for the drawer (if location names are available)
        drawer_location = None
        for loc in positions:
            if drawer_name in loc:
                drawer_location = loc
                break
        # Fallback
        if drawer_location is None:
            drawer_location = "drawer1_location"

        # Find a location for the handle 
        handle_location = None
        if handle_name:
            handle_location = positions.get(handle_name, None)
            # just use the key as location name if string names are used
            handle_location_name = None
            for loc in positions:
                if handle_name in loc:
                    handle_location_name = loc
                    break
        else:
            handle_location_name = None

        # ---- 1. Move robot to the drawer/handle location ---
        # Use execute_go(skill): move from current location to drawer/handle
        try:
            # For symbolic params, we can use keys; for coordinates, pass as required by your skill API
            print(f"[Exploration] Moving to handle/drawer location: {drawer_location}")
            obs, reward, done, info = execute_go(
                env, 
                task, 
                from_location=robot_location, 
                to_location=drawer_location,
                max_steps=50
            )
            robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Failed execute_go: {e}")
            # If execute_go fails, halt exploration
            shutdown_environment(env)
            return

        # ---- 2. Try to pick the handle (execute_pick_handle skill) ---
        if handle_name is not None:
            try:
                print(f"[Exploration] Attempting to pick handle: {handle_name} at {drawer_location}")
                obs, reward, done, info = execute_pick_handle(
                    env, 
                    task, 
                    object_name=handle_name, 
                    location=drawer_location, 
                    max_steps=50
                )
            except Exception as e:
                print(f"[Exploration] Failed execute_pick_handle: {e}")
        else:
            print("[Exploration] No handle found to pick. Exploration ends.")
            shutdown_environment(env)
            return

        # ---- 3. Try to pull the drawer, which will discover if it is locked/unlocked ---
        try:
            print(f"[Exploration] Attempting to pull the drawer: {drawer_name} with handle: {handle_name} at {drawer_location}")
            obs, reward, done, info = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location,
                max_steps=50
            )
            # If successful, (drawer-unlocked drawer1) must be true for success
            print("[Exploration] Pull succeeded: Drawer was unlocked.")
            print("[Result] Missing predicate discovered: (drawer-unlocked {})".format(drawer_name))
        except Exception as e:
            print(f"[Exploration] Pull failed: {e}")
            print("[Result] Drawer may be locked or precondition unmet; missing predicate may relate to locking state.")

        # Exploration ends, proper skill code would continue plan once predicate discovered.
        # Further task-solving using the new knowledge would follow here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()