# run_skeleton_task.py (Completed for Adaptive Exploration)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use skill functions as provided externally

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # This will be required for skill calls
        positions = get_object_positions()

        print("[Exploration] Executing exploration phase to detect missing predicates or knowledge requirements due to planner timeout feedback.")

        # === Exploration Phase ===
        # Suppose we do not know which object or predicate is missing; perform active exploration using available skill primitives.

        # For demonstration, iterate over all known objects and positions, running relevant skill primitives to infer missing predicates.
        # This handles e.g. 'identified', 'weight-known', 'lock-known' predicates via relevant skills.
        # We do NOT define new skill functions, only invoke what is provided.

        # Fetch robot and environment state information
        all_objects = list(positions.keys())
        ready_pose = positions.get("ready-pose", None)
        robot_location = None
        for obj_name, pos in positions.items():
            if obj_name.startswith("robot"):
                robot_location = pos
                break

        # -- Example exploration: try all provided low-level skills on each object/location
        available_skills = [
            "execute_pick",
            "execute_place",
            "execute_push",
            "execute_pull",
            "execute_sweep",
            "execute_rotate",
            "execute_go",
            "execute_gripper"
        ]

        # Maintain state and attempt skills in a robust way
        for obj_name in all_objects:
            # Skip ready-pose and robot, focus on manipulable objects
            if obj_name.startswith("robot") or obj_name == "ready-pose":
                continue

            obj_pos = positions[obj_name]

            # Attempt to use available skills for knowledge exploration; error handling in case of inapplicable actions

            # Try moving robot to object's location (if possible)
            try:
                # Using execute_go to move from robot's location to object's location
                # The skill signature may need source and destination locations
                obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=obj_pos)
                robot_location = obj_pos  # Update current robot location
                print(f"[Exploration] execute_go: Moved robot to {obj_name} at {obj_pos}")
            except Exception as e:
                print(f"[Exploration] execute_go failed for object {obj_name}: {e}")

            # Try to pick the object
            try:
                obs, reward, done = execute_pick(env, task, obj_name, obj_pos)
                print(f"[Exploration] execute_pick: Picked {obj_name}")
            except Exception as e:
                print(f"[Exploration] execute_pick failed for {obj_name}: {e}")

            # Try to pull the object (if applicable: e.g., for handles, drawers)
            try:
                obs, reward, done = execute_pull(env, task, obj_name, obj_pos)
                print(f"[Exploration] execute_pull: Pulled {obj_name}")
            except Exception as e:
                print(f"[Exploration] execute_pull failed for {obj_name}: {e}")

            # Try to push just in case this object is a drawer
            try:
                obs, reward, done = execute_push(env, task, obj_name, obj_pos)
                print(f"[Exploration] execute_push: Pushed {obj_name}")
            except Exception as e:
                print(f"[Exploration] execute_push failed for {obj_name}: {e}")

            # Try to place the object back to initial position or another relevant location (such as inside a drawer)
            try:
                # For placing, assume there is a known drawer location (if exists)
                for drawer_name, drawer_pos in positions.items():
                    if "drawer" in drawer_name:
                        obs, reward, done = execute_place(env, task, obj_name, drawer_name, drawer_pos)
                        print(f"[Exploration] execute_place: Placed {obj_name} in {drawer_name}")
                        break
            except Exception as e:
                print(f"[Exploration] execute_place failed for {obj_name}: {e}")

            # Try sweep, rotate, and gripper skills for completeness
            try:
                obs, reward, done = execute_sweep(env, task, obj_name, obj_pos)
                print(f"[Exploration] execute_sweep: Swept {obj_name}")
            except Exception as e:
                print(f"[Exploration] execute_sweep failed for {obj_name}: {e}")

            try:
                obs, reward, done = execute_rotate(env, task, obj_name, obj_pos)
                print(f"[Exploration] execute_rotate: Rotated {obj_name}")
            except Exception as e:
                print(f"[Exploration] execute_rotate failed for {obj_name}: {e}")

            try:
                obs, reward, done = execute_gripper(env, task)
                print(f"[Exploration] execute_gripper: Actuated gripper for {obj_name}")
            except Exception as e:
                print(f"[Exploration] execute_gripper failed for {obj_name}: {e}")

            # After each round of exploration, check for changes in observation or knowledge (predicate discovery)
            try:
                obs_snapshot = task.get_observation()
                print(f"[Exploration] New observation after acting on {obj_name}: {obs_snapshot}")
            except Exception as e:
                print(f"[Exploration] Observation failed after actions for {obj_name}: {e}")

        print("[Exploration] Exploration phase complete. If a missing predicate needed for planning (such as 'identified', 'weight-known', or 'lock-known') exists, the above skill invocations should reveal it via environment state or observation updates.")

        # -- TASK LOGIC WOULD FOLLOW HERE, after knowledge predicates have been established by exploration --
        # Example: Run the oracle plan using skill invocations provided
        # For demonstration: leave as a placeholder
        # TODO: Replace with the sequence of oracle plan actions for your scenario

        print("[Task] Oracle plan execution phase would proceed here, using the now discovered knowledge/state.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
