# run_skeleton_task.py (Completed Executable with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Only use provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    missing_predicate_found = False
    exploration_results = {}
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === EXPLORATION PHASE to Find Missing Predicate ===
        # We will attempt skills and check their effects to discover predicates

        print("[Exploration] Attempting to find missing predicate using predefined skills.")

        # Step 1: Try identifying objects by moving to all locations (simulate exploration knowledge)
        try:
            locations = []
            for name, pos in positions.items():
                if 'location' in name or 'drawer' in name:
                    locations.append(name)
            # If not available, fallback: aggregate unique positions
            if not locations:
                locations = list(set([v for k, v in positions.items()]))

            robot_current = None
            for name, pos in positions.items():
                if 'robot' in name or 'robby' in name or 'robot-at' in name:
                    robot_current = name
                    break

            # Fallback: pick a location
            if robot_current is None and locations:
                robot_current = locations[0]
            explored_locations = set()
            for loc in locations:
                if loc in explored_locations:
                    continue
                # Use execute_go to each location (simulate execute_go_identify)
                try:
                    print(f"[Exploration] Moving robot to location {loc} to identify objects.")
                    obs, reward, done = execute_go(
                        env, task,
                        from_location=robot_current,
                        to_location=loc
                    )
                    # Mark as explored
                    explored_locations.add(loc)
                    robot_current = loc
                except Exception as e:
                    print(f"[Exploration] Error moving to {loc}: {e}")
        except Exception as ex:
            print(f"[Exploration] Initial movement exploration failed: {ex}")

        # Step 2: Try to pick up objects and see if unknown properties (like weight, durability) get known
        for obj_name in positions:
            if obj_name.startswith('object') or obj_name.startswith('ball') or 'handle' in obj_name:
                # Try picking at all found locations
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying to pick {obj_name} at {loc}...")
                        obs, reward, done = execute_pick(
                            env, task,
                            obj_name, loc
                        )
                        print(f"[Exploration] Picked {obj_name} at {loc}.")
                        # Now, try to place it back (if object, not handle)
                        if not ('handle' in obj_name):
                            for d in locations:
                                try:
                                    print(f"[Exploration] Placing {obj_name} at {d}...")
                                    obs, reward, done = execute_place(
                                        env, task,
                                        obj_name, d, loc
                                    )
                                    print(f"[Exploration] Placed {obj_name} in drawer {d}.")
                                except Exception as e:
                                    # Not all objects/locations will allow place, so continue.
                                    continue
                    except Exception as e:
                        continue

        # Step 3: Try to unlock or interact with drawers using push/pull, check lock-known
        for d_name in positions:
            if 'drawer' in d_name:
                # Find a handle for this drawer
                candidate_handles = [o for o in positions if 'handle' in o]
                if candidate_handles:
                    handle = candidate_handles[0]
                    loc = None
                    for l in locations:
                        if l in d_name or l in handle:
                            loc = l
                            break
                    if loc is None and locations:
                        loc = locations[0]
                    # Try picking the handle
                    try:
                        print(f"[Exploration] Picking handle {handle} at {loc} for drawer {d_name}.")
                        obs, reward, done = execute_pick(
                            env, task,
                            handle, loc
                        )
                        print(f"[Exploration] Picked handle {handle}.")
                        # Try pulling drawer
                        try:
                            print(f"[Exploration] Pulling drawer {d_name} with handle {handle} at {loc}.")
                            obs, reward, done = execute_pull(
                                env, task,
                                d_name, handle, loc
                            )
                            print(f"[Exploration] Pulled drawer {d_name}.")
                        except Exception as e:
                            print(f"[Exploration] Failed to pull drawer {d_name}: {e}")
                    except Exception as e:
                        print(f"[Exploration] Could not pick drawer handle {handle}: {e}")

        # Step 4: Try pushing all drawers
        for d_name in positions:
            if 'drawer' in d_name:
                loc = None
                for l in locations:
                    if l in d_name:
                        loc = l
                        break
                if loc is None and locations:
                    loc = locations[0]
                try:
                    print(f"[Exploration] Pushing drawer {d_name} at {loc}...")
                    obs, reward, done = execute_push(
                        env, task,
                        d_name, loc
                    )
                    print(f"[Exploration] Pushed {d_name} at {loc}.")
                except Exception as e:
                    continue

        print("[Exploration] Exploration Phase Complete. If the planning issue persists, a missing predicate exists in the domain/observation.")
        print("[INFO] Note: Predicates such as 'lock-known', 'identified', or other *-known are candidates. Please check logs or use external visualization tools for further analysis.")

        # END OF EXPLORATION PHASE

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
