# run_skeleton_task.py (Completed for Exploration with Missing Predicate Detection)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use: execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_go, execute_gripper

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase for Missing Predicate Detection === #
        #
        # The purpose: Explore the environment to collect information about objects,
        # specifically to determine which predicate could be missing (as per the feedback).
        #
        # We'll systematically try to use the predefined skills to gather info such as
        # identification, temperature, weight, durability, and lock-state using only existing skills.
        # In this example, we attempt to interact with each relevant object.

        print("[Exploration] Beginning to explore the environment for missing predicate(s)")

        # Attempt to identify objects, check drawers, and try actions with various skills.
        # (Assume that positions returns a dict: object name -> position)
        failed_skills = []
        found_missing_predicate = False
        suspected_predicate = None

        # Example logic: try picking and pulling each object to see if some skill fails or discovers info
        for obj_name, pos in positions.items():
            try:
                # 1. Try identifying the object (simulate `execute_go` to its position)
                print(f"[Exploration] Moving to {obj_name} at position {pos}")
                obs, reward, done = execute_go(env, task, from_pos=None, to_pos=pos)
                if done:
                    print(f"[Exploration] Task ended after moving to {obj_name}!")
                    break

                # 2. Try picking the object
                print(f"[Exploration] Attempting to pick {obj_name}")
                obs, reward, done = execute_pick(env, task, target=pos, timeout=5.0)
                if done:
                    print(f"[Exploration] Successfully picked {obj_name}")
                else:
                    print(f"[Exploration] Could not pick {obj_name}")
                    failed_skills.append(('execute_pick', obj_name))
                
                # 3. Try pulling using the handle (simulate lock-known testing)
                print(f"[Exploration] Attempting to pull {obj_name}")
                try:
                    obs, reward, done = execute_pull(env, task, target=pos, timeout=5.0)
                    if done:
                        print(f"[Exploration] Pulled {obj_name} and possibly unlocked/identified it.")
                    else:
                        print(f"[Exploration] Could not pull {obj_name} (might be locked, or missing lock-known predicate)")
                        failed_skills.append(('execute_pull', obj_name))
                        suspected_predicate = 'lock-known'
                        found_missing_predicate = True
                        break
                except Exception as e:
                    print(f"[Exploration] Exception during pull: {e}")
                    failed_skills.append(('execute_pull_exception', obj_name))
                    suspected_predicate = 'lock-known'
                    found_missing_predicate = True
                    break

            except Exception as e:
                print(f"[Exploration] Exception for object {obj_name}: {e}")
                failed_skills.append(('exception', obj_name))

        # === Report Outcome of Exploration ===
        if found_missing_predicate:
            print(f"[Exploration] Suspected missing predicate is: {suspected_predicate}.")
        else:
            print("[Exploration] No obvious missing predicate found from interactions.")

        if failed_skills:
            print(f"[Exploration] Some skill applications failed: {failed_skills}")

        # === (After Exploration) You could now proceed to main plan if desired (not shown) === #

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()