# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only externally provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, including missing predicate exploration.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers
        init_video_writers(obs)

        # Recording wrappings
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()   # Dictionary mapping object names to positions

        # === Exploration Phase: Search for Missing Predicate ===
        print("[Exploration] Attempting to detect missing predicates by invoking available exploration-like skills.")

        # We'll iterate through available skills and their plausible arguments
        # See if any fails due to missing predicate
        
        missing_predicate = None
        exploration_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For demonstration, we will systematically attempt to apply each exploration skill with mock/real arguments if possible,
        # and handle any resulting failed precondition/missing predicate.
        # Since we do not define new skills, we just use these skills and capture failure results

        # Pseudocode for exploration:
        #   For each skill:
        #       Try to execute with plausible argument from object positions
        #       If any exception is thrown or special return (e.g. missing predicate), capture and report it

        found_info = {}
        for skill_name in exploration_skills:
            try:
                # Get function from skill_code by name
                skill_fn = globals().get(skill_name, None)
                if skill_fn is None:
                    continue

                # Prepare plausible arguments depending on function name
                # These are high-level guesses; in a real integration use the param signature
                if skill_name == 'execute_go':
                    # Move robot from one location to another
                    locs = [k for k, v in positions.items() if ('location' in k or 'drawer' in k or 'floor' in k)]
                    if len(locs) >= 2:
                        from_loc, to_loc = locs[0], locs[1]
                        print(f"[Exploration] Trying {skill_name}({from_loc}, {to_loc})")
                        skill_fn(env, task, from_loc, to_loc)
                elif skill_name == 'execute_pick':
                    # Try to pick up an object on floor at some location
                    objs = [k for k in positions if 'object' in k or 'handle' in k]
                    locs = [k for k, v in positions.items() if ('location' in k or 'drawer' in k or 'floor' in k)]
                    if objs and locs:
                        target = objs[0]
                        location = locs[0]
                        print(f"[Exploration] Trying {skill_name}({target}, {location})")
                        skill_fn(env, task, target, location)
                elif skill_name == 'execute_place':
                    # Try to place object into drawer at location
                    objs = [k for k in positions if 'object' in k]
                    drawers = [k for k in positions if 'drawer' in k]
                    locs = [k for k, v in positions.items() if ('location' in k or 'drawer' in k or 'floor' in k)]
                    if objs and drawers and locs:
                        obj = objs[0]
                        drawer = drawers[0]
                        location = locs[0]
                        print(f"[Exploration] Trying {skill_name}({obj}, {drawer}, {location})")
                        skill_fn(env, task, obj, drawer, location)
                elif skill_name == 'execute_push':
                    # Push a drawer
                    drawers = [k for k in positions if 'drawer' in k]
                    locs = [k for k, v in positions.items() if ('location' in k or 'drawer' in k or 'floor' in k)]
                    if drawers and locs:
                        drawer = drawers[0]
                        location = locs[0]
                        print(f"[Exploration] Trying {skill_name}({drawer}, {location})")
                        skill_fn(env, task, drawer, location)
                elif skill_name == 'execute_pull':
                    # Pull a drawer by handle (simulate grasp)
                    drawers = [k for k in positions if 'drawer' in k]
                    handles = [k for k in positions if 'handle' in k]
                    locs = [k for k, v in positions.items() if ('location' in k or 'drawer' in k or 'floor' in k)]
                    if drawers and handles and locs:
                        drawer = drawers[0]
                        handle = handles[0]
                        location = locs[0]
                        print(f"[Exploration] Trying {skill_name}({drawer}, {handle}, {location})")
                        skill_fn(env, task, drawer, handle, location)
                elif skill_name == 'execute_sweep':
                    # Sweep an object at its location
                    objs = [k for k in positions if 'object' in k]
                    locs = [k for k, v in positions.items() if ('location' in k or 'drawer' in k or 'floor' in k)]
                    if objs and locs:
                        obj = objs[0]
                        location = locs[0]
                        print(f"[Exploration] Trying {skill_name}({obj}, {location})")
                        skill_fn(env, task, obj, location)
                elif skill_name == 'execute_rotate':
                    # In many domains 'rotate' takes object param
                    objs = [k for k in positions if 'object' in k]
                    if objs:
                        obj = objs[0]
                        print(f"[Exploration] Trying {skill_name}({obj})")
                        skill_fn(env, task, obj)
                elif skill_name == 'execute_gripper':
                    # Gripper action, no param
                    print(f"[Exploration] Trying {skill_name}()")
                    skill_fn(env, task)
                # If no error, mark as succeeded
                found_info[skill_name] = "succeeded"
            except Exception as e:
                # If error occurs, record potential missing predicate or other issues
                found_info[skill_name] = f"FAILED: {str(e)}"
                if "predicate" in str(e).lower() or "precondition" in str(e).lower() or "unknown" in str(e).lower():
                    print(f"[Exploration] Detected potential missing predicate via {skill_name}: {e}")
                    missing_predicate = str(e)
        
        print("[Exploration] Skill execution results:", found_info)
        if missing_predicate:
            print(f"[Exploration] Missing predicate likely: {missing_predicate}")
        else:
            print("[Exploration] No missing predicate detected directly during exploration.")

        # === MAIN ORACLE PLAN EXECUTION ===

        # TODO: Place your main oracle plan logic here – execute each plan step using only available skills,
        # following the proper parameterization depending on objects/locations/tasks in your scenario.
        # Example (to be replaced with your real task):
        # For demonstration, let's assume we need to:
        #   1. Move to obj_1 location
        #   2. Pick up obj_1
        #   3. Move to drawer
        #   4. Place obj_1 in drawer

        try:
            obj_names   = [k for k in positions if 'object' in k]
            drawer_names= [k for k in positions if 'drawer' in k]
            loc_names   = [k for k in positions if 'location' in k or 'floor' in k or 'room' in k]
            
            if obj_names and drawer_names and loc_names:
                obj = obj_names[0]
                drawer = drawer_names[0]
                loc_obj = loc_names[0]
                loc_drawer = loc_names[-1] if len(loc_names) > 1 else loc_names[0]
                # Step 1: Move to object location
                print(f"[Plan] Moving to {loc_obj}")
                execute_go(env, task, "ready-pose", loc_obj)
                # Step 2: Pick up object
                print(f"[Plan] Picking up {obj} at {loc_obj}")
                execute_pick(env, task, obj, loc_obj)
                # Step 3: Move to drawer location
                print(f"[Plan] Moving to {loc_drawer}")
                execute_go(env, task, loc_obj, loc_drawer)
                # Step 4: Place object in drawer
                print(f"[Plan] Placing {obj} in {drawer} at {loc_drawer}")
                execute_place(env, task, obj, drawer, loc_drawer)
                print("[Plan] Oracle plan steps executed.")
            else:
                print("[Plan] Not enough objects, drawers, or locations in positions for demo plan.")

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")

        # Task complete

    finally:
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()