# run_skeleton_task.py (Completed Exploration Skeleton Task)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Only use existing primitives such as execute_pick, execute_place, execute_push, etc.

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration to find missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase ===
        # Attempt to execute each available primitive skill to determine the missing predicate
        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # This set will keep track of predicates/features observed to be required/missing
        missing_predicates = set()
        success_skills = []
        failed_skills = []

        print("\n[Exploration] Testing available skills to find missing predicates...")

        # Get a list of objects, drawers, handles, locations by type from positions dict, if possible
        objects = []
        drawers = []
        handles = []
        locations = []
        for name, info in positions.items():
            obj_type = info.get('type', '').lower()
            if 'drawer' in obj_type:
                drawers.append(name)
            if 'handle' in obj_type:
                handles.append(name)
            if 'object' in obj_type and not any([x in obj_type for x in ['drawer','handle']]):
                objects.append(name)
            if 'location' in obj_type or 'pose' in obj_type or 'place' in obj_type:
                locations.append(name)
        # Fallback: just use all non-drawer/handles as locations if not specifically typed
        if not locations:
            for name in positions.keys():
                locations.append(name)
        # Remove duplicates
        locations = list(set(locations))

        # Get current robot location if available in obs (look for 'robot-at ...' or similar)
        robot_location = None
        if 'robot_at' in obs:
            robot_location = obs['robot_at']
        elif len(locations) > 0:
            robot_location = locations[0]  # Fallback

        # Try simple exploration for each skill using arbitrarily picked entities from those found above
        for skill in available_skills:
            try:
                print(f"\n[Exploration] Trying skill: {skill}")
                # Prepare plausible (dummy) arguments based on skill signature
                if skill == 'execute_pick':
                    # Try to pick an object from a location
                    if objects and locations:
                        obj = objects[0]
                        loc = locations[0]
                        res = execute_pick(env, task, obj, loc)
                        success_skills.append(skill)
                        print(f"  > Success: execute_pick({obj}, {loc})")
                    else:
                        raise ValueError("No objects or locations found for execute_pick.")
                elif skill == 'execute_place':
                    # Try to place an object into a drawer from a location
                    if objects and drawers and locations:
                        obj = objects[0]
                        drw = drawers[0]
                        loc = locations[0]
                        res = execute_place(env, task, obj, drw, loc)
                        success_skills.append(skill)
                        print(f"  > Success: execute_place({obj}, {drw}, {loc})")
                    else:
                        raise ValueError("No objects, drawers, or locations found for execute_place.")
                elif skill == 'execute_push':
                    if drawers and locations:
                        drw = drawers[0]
                        loc = locations[0]
                        res = execute_push(env, task, drw, loc)
                        success_skills.append(skill)
                        print(f"  > Success: execute_push({drw}, {loc})")
                    else:
                        raise ValueError("No drawers or locations found for execute_push.")
                elif skill == 'execute_pull':
                    # Try pulling a drawer via handle; pull requires holding the handle object
                    if drawers and handles and locations:
                        drw = drawers[0]
                        hnd = handles[0]
                        loc = locations[0]
                        # Assume need to pick handle first
                        try:
                            execute_pick(env, task, hnd, loc)
                        except Exception:
                            pass
                        res = execute_pull(env, task, drw, hnd, loc)
                        success_skills.append(skill)
                        print(f"  > Success: execute_pull({drw}, {hnd}, {loc})")
                    else:
                        raise ValueError("No drawers, handles, or locations found for execute_pull.")
                elif skill == 'execute_sweep':
                    if objects and locations:
                        obj = objects[0]
                        loc = locations[0]
                        res = execute_sweep(env, task, obj, loc)
                        success_skills.append(skill)
                        print(f"  > Success: execute_sweep({obj}, {loc})")
                    else:
                        raise ValueError("No objects or locations found for execute_sweep.")
                elif skill == 'execute_rotate':
                    # Not enough info, just call with no arg
                    res = execute_rotate(env, task)
                    success_skills.append(skill)
                    print(f"  > Success: execute_rotate()")
                elif skill == 'execute_go':
                    if locations and len(locations) > 1:
                        from_loc = locations[0]
                        to_loc = locations[1]
                        res = execute_go(env, task, from_loc, to_loc)
                        success_skills.append(skill)
                        print(f"  > Success: execute_go({from_loc}, {to_loc})")
                    elif locations:
                        print(f"  > Skipped execute_go (not enough distinct locations)")
                    else:
                        raise ValueError("No locations found for execute_go.")
                elif skill == 'execute_gripper':
                    res = execute_gripper(env, task)
                    success_skills.append(skill)
                    print(f"  > Success: execute_gripper()")
                else:
                    print(f"  > Skipped unknown skill: {skill}")
            except Exception as e:
                failed_skills.append(skill)
                msg = str(e)
                # Naive logic: parse error for likely missing predicate or precondition
                if ("precondition" in msg or "predicate" in msg or "failed" in msg):
                    missing_predicates.add(msg)
                print(f"  > Failed: {skill}: {msg}")

        print("\n[Exploration] Skills executed. Summary:")
        print(f"  Skills succeeded: {success_skills}")
        print(f"  Skills failed: {failed_skills}")
        print(f"  Candidate missing predicates (from errors):")
        for pred in missing_predicates:
            print(f"   - {pred}")

        print("[Exploration] Finished. Use this feedback to adjust the planning domain/problem.")
        
        # Further task/plan execution would go here, e.g. after finding predicates

        # Optionally, implement logic to execute main oracle (if available) plan,
        # Using the skills above (as shown in the skeleton)
        #
        # print("[Task] [Oracle Plan Execution] ... (not implemented in this exploration scaffold)")
        #

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
