# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration phase for missing predicate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # positions: dict, e.g., {'obj1': (x,y,z), ...}

        # === Exploration Phase ===
        # Goal: Determine what predicate is missing
        #
        # Strategy: Try each exploration skill from the provided knowledge, see if the environment responds or errors.
        # Map exploration actions to the available primitives for this simulated generic task

        # Let's collect potential object/location names (will gracefully handle KeyError)
        object_names = list(positions.keys())
        if len(object_names) == 0:
            print("No objects detected in environment, cannot proceed with exploration.")
            return

        # We'll try all available skills as a probe sequence for learning "missing predicates"
        available_skills = ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        exploration_result = {}

        # For simplicity, pick the first two objects as handle and regular object, and extract all location names
        # (This would be replaced by more systematic type inference if needed)
        possible_locations = set()
        for v in positions.values():
            if isinstance(v, dict) and 'location' in v:
                possible_locations.add(v['location'])
            elif isinstance(v, (tuple, list)) and len(v) >= 3:
                possible_locations.add(tuple(v))

        object1 = object_names[0]
        object2 = object_names[1] if len(object_names) > 1 else object1

        # Try moving the robot to each location using 'execute_go' as a simple exploration
        current_robot_location = None
        try:
            obs_dict = obs if isinstance(obs, dict) else {}
            for k in obs_dict:
                if k.startswith('robot_position') or k == 'robot':
                    current_robot_location = obs_dict[k]
                    break
        except Exception:
            current_robot_location = None

        if current_robot_location is None:
            # Use a default or just one known object location as start
            current_robot_location = positions[object1] if object1 in positions else (0,0,0)

        print("[Exploration] Starting exploration to identify environment predicates...")
        for skill in available_skills:
            try:
                # For predicate learning, attempt each skill and note the result.
                # For realistic exploration, only call a sample (e.g., 'execute_pull'/'execute_push') on a test object+location.
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_go':
                    # Try to move between two locations if available
                    targets = list(possible_locations)
                    if len(targets) >= 2:
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_location=targets[0],
                            to_location=targets[1],
                            max_steps=100,
                        )
                        exploration_result[skill] = 'success'
                elif skill == 'execute_pick':
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name=object1,
                        at_location=positions[object1],
                        max_steps=100,
                    )
                    exploration_result[skill] = 'success'
                elif skill == 'execute_place':
                    # Try placing object1 into (what we guess might be) object2
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj_name=object1,
                        container_name=object2,
                        at_location=positions[object2],
                        max_steps=100,
                    )
                    exploration_result[skill] = 'success'
                elif skill == 'execute_pull':
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        obj_name=object2,
                        at_location=positions[object2],
                        max_steps=100,
                    )
                    exploration_result[skill] = 'success'
                elif skill == 'execute_push':
                    obs, reward, done = execute_push(
                        env,
                        task,
                        obj_name=object2,
                        at_location=positions[object2],
                        max_steps=100,
                    )
                    exploration_result[skill] = 'success'
                elif skill == 'execute_sweep':
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        obj_name=object1,
                        at_location=positions[object1],
                        max_steps=100,
                    )
                    exploration_result[skill] = 'success'
                elif skill == 'execute_rotate':
                    obs, reward, done = execute_rotate(
                        env,
                        task,
                        obj_name=object1,
                        at_location=positions[object1],
                        max_steps=100,
                    )
                    exploration_result[skill] = 'success'
                elif skill == 'execute_gripper':
                    obs, reward, done = execute_gripper(
                        env,
                        task,
                        max_steps=100,
                    )
                    exploration_result[skill] = 'success'
                else:
                    print(f"[Exploration] Unrecognized skill: {skill}")
                    exploration_result[skill] = 'unknown'
            except Exception as e:
                print(f"[Exploration] Exception executing {skill}: {e}")
                exploration_result[skill] = f'error: {str(e)}'

        print("[Exploration] Exploration phase completed.")
        print("[Exploration] Skill execution results:")
        for skill, result in exploration_result.items():
            print(f"    {skill}: {result}")

        # === End of Exploration Phase ===
        # Next, you would analyze these results (exceptions and successes) and infer the missing predicate
        # (This code does not do the learning itself, just demonstrates probe calls as required.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()