from Object_list import Object
from skill_mapping import RobotController


def Long_horizon_task2_oracle_seq() -> None:
    """
    Oracle sequence for the task:
      1.  go  (ready-pose → handle_top_pose)
      2.  pick handle
      3.  pull drawer open
      4.  pick trash from drawer
      5.  place trash into bin
      6.  pick dice from floor
      7.  place dice inside drawer
      8.  push drawer closed
      9.  go  (drawer_front → ready-pose)

    Every step uses ONLY the predefined skills exposed by
    RobotController.execute_movement().
    """

    robot = RobotController()

    # ------------------------------------------------------------------
    # STEP-1 : execute_go  (ready-pose → drawer_top_handle_pose)
    # ------------------------------------------------------------------
    #
    # execute_go is triggered with mode = 7.
    # A target pose must still be provided (pose0), so we pass the
    # drawer-top handle’s pose.  If the object map happens to miss a
    # ready-pose entry we simply rely on the controller’s internal
    # state (go_to_ready_pose later).
    #
    try:
        robot.execute_movement(Object["drawer_top_handle"], mode=7, axis=0)
    except Exception as err:
        print(f"[WARN] execute_go failed (step-1) → {err}")

    # ------------------------------------------------------------------
    # STEP-2 : execute_pick  (handle)
    #  – approach along x-axis (axis = 0)
    # ------------------------------------------------------------------
    robot.execute_movement(Object["drawer_top_handle"], mode=1, axis=0)

    # ------------------------------------------------------------------
    # STEP-3 : execute_pull  (open drawer)
    # ------------------------------------------------------------------
    robot.execute_movement(
        Object["drawer_top_handle"], mode=4, axis=0, distance=0.12
    )

    # ------------------------------------------------------------------
    # STEP-4 : execute_pick  (trash inside drawer)
    #  – approach vertically (axis = 2)
    # ------------------------------------------------------------------
    robot.execute_movement(Object["trash"], mode=1, axis=2)

    # ------------------------------------------------------------------
    # STEP-5 : execute_place  (trash → bin)
    # ------------------------------------------------------------------
    robot.execute_movement(mode=8)  # go_to_ready_pose (need-ready → false)
    robot.execute_movement(Object["trash_bin"], mode=2, axis=2)

    # ------------------------------------------------------------------
    # STEP-6 : execute_pick  (dice on floor)
    # ------------------------------------------------------------------
    robot.execute_movement(mode=8)  # ready again before next pick
    robot.execute_movement(Object["dice1"], mode=1, axis=2)

    # ------------------------------------------------------------------
    # STEP-7 : execute_place  (dice → inside drawer)
    # ------------------------------------------------------------------
    robot.execute_movement(mode=8)
    # Use a default drawer placing pose; fallback to handle if missing.
    drawer_place_pose = Object.get("drawer_top_place_left",
                                   Object["drawer_top_handle"])
    robot.execute_movement(drawer_place_pose, mode=2, axis=2)

    # ------------------------------------------------------------------
    # STEP-8 : execute_push  (close drawer)
    # ------------------------------------------------------------------
    robot.execute_movement(mode=8)
    robot.execute_movement(
        Object["drawer_top_handle"], mode=3, axis=0, distance=0.12
    )

    # ------------------------------------------------------------------
    # STEP-9 : execute_go  (drawer_front → ready-pose)
    # ------------------------------------------------------------------
    # If ready_pose exists in the map, use execute_go (mode-7);
    # otherwise simply call go_to_ready_pose (mode-8).
    ready_pose = Object.get("ready_pose", None)
    if ready_pose is not None:
        try:
            robot.execute_movement(ready_pose, mode=7, axis=0)
        except Exception as err:
            print(f"[WARN] execute_go failed (step-9) → {err}")
            robot.execute_movement(mode=8)
    else:
        robot.execute_movement(mode=8)


def main() -> None:
    Long_horizon_task2_oracle_seq()


if __name__ == "__main__":
    main()