# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must execute the "execute_go_ready" skill to clear it.

        # For demonstration, let's assume a generic plan:
        # 1. Move to an object location
        # 2. Pick the object (sets need-ready)
        # 3. Go to ready pose (clears need-ready)
        # 4. Move to a drawer
        # 5. Place the object in the drawer (sets need-ready)
        # 6. Go to ready pose (clears need-ready)
        # 7. Push the drawer (does not set need-ready)
        # (The actual plan may differ, but this structure ensures we handle need-ready correctly.)

        # === Example: Find an object on the floor and a drawer ===
        # For this example, we will search for an object and a drawer in the positions dict.
        object_name = None
        object_pos = None
        drawer_name = None
        drawer_pos = None
        ready_pose = None

        for name, pos in positions.items():
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
                drawer_pos = pos
            elif 'object' in name and object_name is None:
                object_name = name
                object_pos = pos
            elif 'ready' in name:
                ready_pose = pos

        if object_name is None or drawer_name is None or ready_pose is None:
            print("[Error] Could not find required objects or ready pose in the environment.")
            return

        # === 1. Move to object location ===
        print(f"[Task] Moving to object {object_name} at {object_pos}")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_pos=positions.get('robot', (0,0,0)),
                to_pos=object_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] execute_go failed: {e}")
            return

        # === 2. Pick the object (sets need-ready) ===
        print(f"[Task] Picking object {object_name}")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                target_obj=object_name,
                target_pos=object_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] execute_pick failed: {e}")
            return

        # === 3. Go to ready pose (clears need-ready) ===
        print(f"[Task] Going to ready pose to clear need-ready")
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_pos=object_pos,
                ready_pos=ready_pose,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] execute_go_ready failed: {e}")
            return

        # === 4. Move to drawer location ===
        print(f"[Task] Moving to drawer {drawer_name} at {drawer_pos}")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_pos=ready_pose,
                to_pos=drawer_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] execute_go (to drawer) failed: {e}")
            return

        # === 5. Place the object in the drawer (sets need-ready) ===
        print(f"[Task] Placing object {object_name} into drawer {drawer_name}")
        try:
            obs, reward, done = execute_place(
                env,
                task,
                target_obj=object_name,
                target_pos=drawer_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] execute_place failed: {e}")
            return

        # === 6. Go to ready pose (clears need-ready) ===
        print(f"[Task] Going to ready pose to clear need-ready after placing")
        try:
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_pos=drawer_pos,
                ready_pos=ready_pose,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] execute_go_ready (after place) failed: {e}")
            return

        # === 7. Push the drawer closed (if needed) ===
        print(f"[Task] Pushing drawer {drawer_name} closed")
        try:
            obs, reward, done = execute_push(
                env,
                task,
                target_drawer=drawer_name,
                target_pos=drawer_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] execute_push failed: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
