# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that may block further actions.
        # We must ensure that after actions that set (need-ready), we call the appropriate skill to clear it.
        # In the provided domain, execute_go_ready clears (need-ready).
        # So, after any action that sets (need-ready), we must call execute_go_ready before proceeding.

        # === Example Plan Execution ===
        # For demonstration, we simulate a generic plan with exploration for (need-ready).
        # In a real scenario, you would parse the oracle plan and execute each step accordingly.

        # Example: Suppose the plan is to pick an object from the floor, then place it in a drawer.
        # The pick action sets (need-ready), so we must call execute_go_ready before the next action.

        # For this example, let's assume:
        # - object_name: 'obj1'
        # - drawer_name: 'drawer1'
        # - handle_name: 'handle1'
        # - pick_location: 'loc1'
        # - drawer_location: 'loc2'
        # - ready_pose: 'ready-pose'

        # These names should be replaced with actual names from your environment/observation.
        object_name = None
        drawer_name = None
        handle_name = None
        pick_location = None
        drawer_location = None
        ready_pose = 'ready-pose'

        # Try to infer object and location names from positions dict
        # This is a placeholder; adapt as needed for your environment
        for k in positions:
            if 'obj' in k and object_name is None:
                object_name = k
            if 'drawer' in k and drawer_name is None:
                drawer_name = k
            if 'handle' in k and handle_name is None:
                handle_name = k
            if 'loc' in k and pick_location is None:
                pick_location = k
            if 'loc' in k and drawer_location is None and k != pick_location:
                drawer_location = k

        # Fallbacks if not found
        if object_name is None:
            object_name = 'object_1'
        if drawer_name is None:
            drawer_name = 'drawer_1'
        if handle_name is None:
            handle_name = 'handle_1'
        if pick_location is None:
            pick_location = 'location_1'
        if drawer_location is None:
            drawer_location = 'location_2'

        # Track if (need-ready) is set
        need_ready = False

        # 1. Move to pick location
        try:
            print(f"[Task] Moving to pick location: {pick_location}")
            obs, reward, done = execute_go(env, task, from_location=ready_pose, to_location=pick_location)
        except Exception as e:
            print(f"[Error] execute_go failed: {e}")

        # 2. Pick object from floor (sets need-ready)
        try:
            print(f"[Task] Picking object: {object_name} at {pick_location}")
            obs, reward, done = execute_pick(env, task, object_name, pick_location)
            need_ready = True
        except Exception as e:
            print(f"[Error] execute_pick failed: {e}")

        # 3. If need-ready, go to ready pose to clear it
        if need_ready:
            try:
                print(f"[Task] Going to ready pose to clear (need-ready)")
                obs, reward, done = execute_go_ready(env, task, from_location=pick_location)
                need_ready = False
            except Exception as e:
                print(f"[Error] execute_go_ready failed: {e}")

        # 4. Move to drawer location
        try:
            print(f"[Task] Moving to drawer location: {drawer_location}")
            obs, reward, done = execute_go(env, task, from_location=ready_pose, to_location=drawer_location)
        except Exception as e:
            print(f"[Error] execute_go failed: {e}")

        # 5. Place object into drawer (sets need-ready)
        try:
            print(f"[Task] Placing object: {object_name} into drawer: {drawer_name} at {drawer_location}")
            obs, reward, done = execute_place(env, task, object_name, drawer_name, drawer_location)
            need_ready = True
        except Exception as e:
            print(f"[Error] execute_place failed: {e}")

        # 6. If need-ready, go to ready pose to clear it
        if need_ready:
            try:
                print(f"[Task] Going to ready pose to clear (need-ready)")
                obs, reward, done = execute_go_ready(env, task, from_location=drawer_location)
                need_ready = False
            except Exception as e:
                print(f"[Error] execute_go_ready failed: {e}")

        # === End of Plan ===

        print("[Task] Plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
