# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()

        # ---- Begin Exploration Phase for Missing Predicate ----
        #
        # As per feedback, we're missing information/predicate regarding the lock state
        # of a drawer or its handle (e.g., (drawer-unlocked drawer_top_handle)).
        # So, here we perform an exploration action to discover this missing state.
        #
        # The appropriate primitive from the available skills is 'execute_pull'
        # (which, in the exploration domain, can be used to update knowledge of lock status).
        #
        # We assume that, during exploration, a handle object on the drawer must be pulled
        # to check if it is locked/unlocked.

        # Try to auto-detect a likely handle and drawer for exploration based on positions
        # and names (typical RLBench/Robosuite naming conventions).
        handle_name, drawer_name, handle_pos, drawer_pos, robot_pos = None, None, None, None, None
        for k in positions.keys():
            if 'handle' in k and (handle_name is None):
                handle_name = k
                handle_pos = positions[k]
            if 'drawer' in k and not 'handle' in k and (drawer_name is None):
                drawer_name = k
                drawer_pos = positions[k]
        for k in positions.keys():
            if 'robot' in k or 'arm' in k:
                robot_pos = positions[k]

        # Fallback to default names if not found
        if handle_name is None:
            handle_name = 'drawer_top_handle'
        if drawer_name is None:
            drawer_name = 'drawer_top'

        # Location for the robot (fallback to initial pose if none found)
        if robot_pos is None and hasattr(task, 'robot_init_pos'):
            robot_pos = task.robot_init_pos
        elif robot_pos is None:
            robot_pos = positions.get('robot', None)

        print(f"[Exploration] Trying to pull handle {handle_name} to learn lock state on {drawer_name}")
        try:
            # 1. Move to handle position if required (with execute_go)
            # We'll use the specialized 'execute_go' skill, from robot's current (ready) position to handle's position
            pre_pose = 'ready-pose'
            handle_location = 'drawer_location' if 'drawer_location' in positions else handle_pos

            # First, go to handle location (Skip if already there)
            obs, reward, done = execute_go(env, task, pre_pose, handle_location)
            print(f"[Exploration] Robot moved to handle location: {handle_location}")

            # 2. Grasp the handle (using execute_pick, which is typically valid for handles)
            obs, reward, done = execute_pick(env, task, handle_name, handle_location)
            print(f"[Exploration] Robot picked up handle: {handle_name}")

            # 3. Attempt to pull (using execute_pull)
            # This will either succeed, or (importantly for exploration) establish lock/unlock knowledge.
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, handle_location)
            print(f"[Exploration] Robot attempted to pull handle '{handle_name}' to test if '{drawer_name}' is locked.")

        except Exception as e:
            print(f"[Exploration] Exception during lock exploration: {e}")

        # ---- End Exploration Phase ----

        # After exploration, proceed with the full oracle plan (not included here, since only exploration is needed)
        # Insert here any further plan steps after exploration, e.g., manipulating objects, opening/placing/closing

        # Example of where further steps would go (after the missing predicate becomes available):
        # obs, reward, done = execute_pick(...)
        # obs, reward, done = execute_place(...)
        # etc.

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
