# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that may block plan execution.
        # We must ensure that after actions that set (need-ready), we call the appropriate skill to clear it.
        # In the provided domain, execute_go_ready is the only action that clears (need-ready).
        # So, after any action that sets (need-ready), we must call execute_go_ready.

        # === Example Plan Execution ===
        # For demonstration, we simulate a generic plan with exploration for (need-ready).
        # In a real scenario, the plan would be generated by a planner and parsed step-by-step.

        # For this example, let's assume the following sequence:
        # 1. Move to object location
        # 2. Pick object (sets need-ready)
        # 3. Go to ready pose (clears need-ready)
        # 4. Move to drawer
        # 5. Place object in drawer (sets need-ready)
        # 6. Go to ready pose (clears need-ready)
        # 7. Push drawer closed (does not set need-ready)
        # 8. Task complete

        # The actual object and location names should be replaced with those from your environment.
        # We'll use placeholders here; replace with actual names as needed.

        try:
            # Example object and location names (replace with actual names from your environment)
            object_name = 'object_1'
            object_location = 'location_a'
            drawer_name = 'drawer_1'
            drawer_location = 'location_b'

            # 1. Move to object location
            print(f"[Task] Moving to {object_location} to approach {object_name}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',  # Assuming robot starts at ready-pose
                to_location=object_location
            )
            if done:
                print("[Task] Task ended after move!")
                return

            # 2. Pick object (sets need-ready)
            print(f"[Task] Picking up {object_name} at {object_location}")
            obs, reward, done = execute_pick(
                env,
                task,
                obj=object_name,
                location=object_location
            )
            if done:
                print("[Task] Task ended after pick!")
                return

            # 3. Go to ready pose (clears need-ready)
            print("[Task] Going to ready-pose to clear need-ready")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=object_location
            )
            if done:
                print("[Task] Task ended after go_ready!")
                return

            # 4. Move to drawer location
            print(f"[Task] Moving to {drawer_location} to approach {drawer_name}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location=drawer_location
            )
            if done:
                print("[Task] Task ended after move to drawer!")
                return

            # 5. Place object in drawer (sets need-ready)
            print(f"[Task] Placing {object_name} into {drawer_name} at {drawer_location}")
            obs, reward, done = execute_place(
                env,
                task,
                obj=object_name,
                drawer=drawer_name,
                location=drawer_location
            )
            if done:
                print("[Task] Task ended after place!")
                return

            # 6. Go to ready pose (clears need-ready)
            print("[Task] Going to ready-pose to clear need-ready after place")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_location
            )
            if done:
                print("[Task] Task ended after go_ready!")
                return

            # 7. Push drawer closed (does not set need-ready)
            print(f"[Task] Pushing {drawer_name} closed at {drawer_location}")
            obs, reward, done = execute_push(
                env,
                task,
                drawer=drawer_name,
                location=drawer_location
            )
            if done:
                print("[Task] Task ended after push!")
                return

            print("[Task] Task completed successfully.")

        except Exception as e:
            print(f"[Error] Exception during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
