# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # Feedback indicates (need-ready) is a missing predicate that may block further actions.
        # We will attempt to execute a skill, and if it fails due to (need-ready), we will call execute_go_ready.

        # For demonstration, let's assume we want to pick up an object from the floor.
        # We'll try to pick the first available object on the floor.

        # Find an object on the floor and its location
        object_on_floor = None
        object_pos = None
        for obj_name, pos_info in positions.items():
            if 'on_floor' in pos_info and pos_info['on_floor']:
                object_on_floor = obj_name
                object_pos = pos_info['position']
                break

        # If no object found, just pick any object for demonstration
        if object_on_floor is None:
            for obj_name, pos_info in positions.items():
                if 'position' in pos_info:
                    object_on_floor = obj_name
                    object_pos = pos_info['position']
                    break

        # Assume we know the robot's current location
        robot_location = None
        for name, pos_info in positions.items():
            if name == 'robot' and 'location' in pos_info:
                robot_location = pos_info['location']
                break
        if robot_location is None:
            robot_location = 'ready-pose'  # fallback

        # Try to execute a pick action, handle (need-ready) if encountered
        done = False
        tried_go_ready = False
        while not done:
            try:
                print(f"[Exploration] Attempting to pick {object_on_floor} at {object_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj=object_on_floor,
                    target_pos=object_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Exploration] Pick executed successfully.")
                break
            except Exception as e:
                # Check if the error is due to (need-ready) predicate
                error_msg = str(e)
                if 'need-ready' in error_msg or 'NEED_READY' in error_msg or 'need_ready' in error_msg:
                    if not tried_go_ready:
                        print("[Exploration] Detected (need-ready) blocking action. Executing execute_go_ready.")
                        try:
                            obs, reward, _ = execute_go_ready(
                                env,
                                task,
                                from_location=robot_location,
                                timeout=10.0
                            )
                            tried_go_ready = True
                            print("[Exploration] execute_go_ready executed. Retrying pick.")
                            continue
                        except Exception as e2:
                            print(f"[Exploration] Failed to execute_go_ready: {e2}")
                            break
                    else:
                        print("[Exploration] Already tried execute_go_ready, but still blocked. Aborting.")
                        break
                else:
                    print(f"[Exploration] Exception during pick: {e}")
                    break

        # === Main Task Plan (Example) ===
        # After exploration, you can proceed with the rest of your oracle plan.
        # For demonstration, let's try to place the object into a bin if holding it.

        # Check if holding the object (this would be tracked in your state, here we just try)
        try:
            print(f"[Task] Attempting to place {object_on_floor} into bin.")
            obs, reward, done = execute_place(
                env,
                task,
                target_obj=object_on_floor,
                target_pos='bin',
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print("[Task] Place executed successfully.")
        except Exception as e:
            print(f"[Task] Exception during place: {e}")

        # You can continue with further plan steps as needed, always checking for (need-ready) and calling execute_go_ready if blocked.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
