# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (holding handle_top) is not being achieved; need to explore which predicate is missing.
        # According to exploration knowledge, lock-known is a predicate that can be discovered via execute_pull.
        # We will attempt to use the available skills to explore the environment and check for missing predicates.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For this exploration, we will:
        # 1. Move to the location of the handle (handle_top)
        # 2. Pick up the handle (execute_pick)
        # 3. Attempt to pull (execute_pull) to see if lock-known is discovered

        # --- Step 1: Find handle_top and its location ---
        handle_name = 'handle_top'
        handle_location = None
        for obj_name, pos in positions.items():
            if 'handle' in obj_name:
                handle_name = obj_name
                handle_location = pos
                break

        if handle_location is None:
            print("[Exploration] Could not find handle object in positions. Exploration aborted.")
            return

        print(f"[Exploration] Found handle object: {handle_name} at {handle_location}")

        # --- Step 2: Move robot to handle location ---
        try:
            # Assume robot's current location is available in positions or obs
            robot_location = None
            for obj_name, pos in positions.items():
                if 'robot' in obj_name or 'base' in obj_name:
                    robot_location = pos
                    break
            if robot_location is None:
                # Fallback: try to get from obs
                robot_location = obs.get('robot_pos', None)
            if robot_location is None:
                print("[Exploration] Could not determine robot's current location. Using default 'ready-pose'.")
                robot_location = 'ready-pose'

            print(f"[Exploration] Moving robot from {robot_location} to handle location.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=handle_location
            )
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")

        # --- Step 3: Pick up the handle ---
        try:
            print(f"[Exploration] Attempting to pick up handle: {handle_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=handle_location
            )
        except Exception as e:
            print(f"[Exploration] Exception during execute_pick: {e}")

        # --- Step 4: Attempt to pull the handle (to discover lock-known) ---
        try:
            # Find the drawer associated with the handle
            drawer_name = None
            for obj_name in positions.keys():
                if 'drawer' in obj_name:
                    drawer_name = obj_name
                    break
            if drawer_name is None:
                print("[Exploration] Could not find drawer object. Skipping pull.")
            else:
                print(f"[Exploration] Attempting to pull drawer: {drawer_name} using handle: {handle_name}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer_name,
                    handle_name=handle_name,
                    location=handle_location
                )
        except Exception as e:
            print(f"[Exploration] Exception during execute_pull: {e}")

        # --- Step 5: Check if lock-known predicate is now satisfied ---
        # This would typically be checked via the environment's state or observation.
        # For demonstration, print out the relevant observation keys.
        print("[Exploration] Checking for lock-known predicate in observation/state.")
        if hasattr(obs, 'get'):
            for k in obs.keys():
                if 'lock' in k or 'known' in k:
                    print(f"  {k}: {obs[k]}")
        else:
            print("[Exploration] Observation does not support key lookup.")

        print("[Exploration] Exploration phase complete. If lock-known is now true, the missing predicate has been discovered.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
