# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will attempt to execute available skills in a diagnostic sequence to explore the environment
        # and infer which predicate is missing, using only the available skills.

        # Available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # 1. Try to move the robot to all known locations using execute_go
        # 2. Try to pick up all objects using execute_pick
        # 3. Try to pull/push/rotate as appropriate
        # 4. Observe which actions fail and log the error to infer missing predicates

        # For demonstration, we will attempt to:
        # - Move to each location
        # - Pick each object
        # - Try to pull/push/rotate if possible

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('location_')]
        location_names = [name for name in positions if name.startswith('location_')]

        # If location names are not in the form 'location_x', try to infer from positions
        if not location_names:
            # Try to infer locations from object positions
            location_names = list(set([positions[obj]['location'] for obj in positions if 'location' in positions[obj]]))

        # Track which actions fail and why
        exploration_log = []

        # Try moving to each location
        for loc in location_names:
            try:
                # Assume robot starts at some initial location; get current robot location
                # For this skeleton, we assume the robot starts at the first location
                current_loc = location_names[0]
                if loc != current_loc:
                    print(f"[Exploration] Trying to move from {current_loc} to {loc}")
                    obs, reward, done = execute_go(env, task, from_location=current_loc, to_location=loc)
                    current_loc = loc
            except Exception as e:
                exploration_log.append(f"execute_go to {loc} failed: {e}")

        # Try picking up each object at its location
        for obj in object_names:
            try:
                obj_info = positions[obj]
                obj_loc = obj_info['location'] if 'location' in obj_info else None
                if obj_loc is not None:
                    print(f"[Exploration] Trying to pick {obj} at {obj_loc}")
                    obs, reward, done = execute_pick(env, task, object_name=obj, location=obj_loc)
            except Exception as e:
                exploration_log.append(f"execute_pick {obj} failed: {e}")

        # Try pulling, pushing, rotating each object if possible
        for obj in object_names:
            try:
                obj_info = positions[obj]
                obj_loc = obj_info['location'] if 'location' in obj_info else None
                # Try pull
                print(f"[Exploration] Trying to pull {obj} at {obj_loc}")
                obs, reward, done = execute_pull(env, task, object_name=obj, location=obj_loc)
            except Exception as e:
                exploration_log.append(f"execute_pull {obj} failed: {e}")
            try:
                # Try push
                print(f"[Exploration] Trying to push {obj} at {obj_loc}")
                obs, reward, done = execute_push(env, task, object_name=obj, location=obj_loc)
            except Exception as e:
                exploration_log.append(f"execute_push {obj} failed: {e}")
            try:
                # Try rotate
                print(f"[Exploration] Trying to rotate {obj} at {obj_loc}")
                obs, reward, done = execute_rotate(env, task, object_name=obj, location=obj_loc)
            except Exception as e:
                exploration_log.append(f"execute_rotate {obj} failed: {e}")

        # Try sweep and gripper actions as no-ops
        for obj in object_names:
            try:
                obj_info = positions[obj]
                obj_loc = obj_info['location'] if 'location' in obj_info else None
                print(f"[Exploration] Trying to sweep {obj} at {obj_loc}")
                obs, reward, done = execute_sweep(env, task, object_name=obj, location=obj_loc)
            except Exception as e:
                exploration_log.append(f"execute_sweep {obj} failed: {e}")

        try:
            print(f"[Exploration] Trying gripper action")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            exploration_log.append(f"execute_gripper failed: {e}")

        # Print exploration log to help identify missing predicates
        print("=== Exploration Log ===")
        for log_entry in exploration_log:
            print(log_entry)
        print("=======================")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate from the exploration log,
        # update the domain/problem PDDL or inform the user to add the missing predicate.

        # The rest of the code would execute the actual oracle plan using the available skills,
        # but since the plan cannot be generated due to the missing predicate, we stop here.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()