# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at table1)
        # The predicate (robot-at table1) is present in the observation, but the domain expects (robot-at ?p - location)
        # or, in the exploration domain, (robot-at ?r - robot ?loc - location).
        # The missing predicate is likely related to the robot's location or the identification of objects at a location.

        # We will use the available skills to explore the environment and try to identify the missing predicate.
        # Since only predefined skills can be used, and exploration skills are not in the available skill list,
        # we use the closest available skills: execute_go, execute_pick, execute_place, execute_push, execute_pull, execute_sweep, execute_rotate, execute_gripper

        # For exploration, we will attempt to move the robot to 'table1' (from its current location), and try to interact with objects there.
        # This will help us determine if the predicate (robot-at table1) is sufficient, or if further identification is needed.

        # Step 1: Find robot's current location and all locations
        robot_location = None
        all_locations = []
        for key in positions:
            if key.startswith('robot'):
                robot_location = positions[key]['location']
            if positions[key].get('type', None) == 'location':
                all_locations.append(key)
        # Fallback: If not found, use a default
        if robot_location is None:
            robot_location = 'ready-pose'
        if not all_locations:
            all_locations = ['table1', 'drawer1', 'bin', 'ready-pose']

        # Step 2: Try to move to table1 using execute_go
        try:
            print("[Exploration] Moving robot from", robot_location, "to table1")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location='table1'
            )
            print("[Exploration] Robot moved to table1.")
        except Exception as e:
            print("[Exploration] Failed to move robot to table1:", e)

        # Step 3: Try to pick any object on table1 using execute_pick
        # Find objects at table1
        objects_on_table1 = []
        for obj_name, info in positions.items():
            if info.get('location', None) == 'table1' and info.get('type', None) == 'object':
                objects_on_table1.append(obj_name)
        for obj in objects_on_table1:
            try:
                print(f"[Exploration] Attempting to pick {obj} at table1")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=obj,
                    location='table1'
                )
                print(f"[Exploration] Picked {obj} at table1.")
                break  # Only pick one for exploration
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj} at table1:", e)

        # Step 4: Try to place the object in the bin using execute_place
        if objects_on_table1:
            try:
                print(f"[Exploration] Attempting to place {objects_on_table1[0]} into bin")
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=objects_on_table1[0],
                    location='bin'
                )
                print(f"[Exploration] Placed {objects_on_table1[0]} into bin.")
            except Exception as e:
                print(f"[Exploration] Failed to place {objects_on_table1[0]} into bin:", e)

        # Step 5: Try to push or pull drawer if present
        for loc in all_locations:
            if 'drawer' in loc:
                try:
                    print(f"[Exploration] Attempting to pull {loc}")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name=loc,
                        handle_name=f"{loc}_handle",
                        location=loc
                    )
                    print(f"[Exploration] Pulled {loc}.")
                except Exception as e:
                    print(f"[Exploration] Failed to pull {loc}:", e)
                try:
                    print(f"[Exploration] Attempting to push {loc}")
                    obs, reward, done = execute_push(
                        env,
                        task,
                        drawer_name=loc,
                        location=loc
                    )
                    print(f"[Exploration] Pushed {loc}.")
                except Exception as e:
                    print(f"[Exploration] Failed to push {loc}:", e)

        # Step 6: Try to sweep or rotate any object on the floor
        for obj_name, info in positions.items():
            if info.get('on_floor', False):
                try:
                    print(f"[Exploration] Attempting to sweep {obj_name} at {info.get('location', 'unknown')}")
                    obs, reward, done = execute_sweep(
                        env,
                        task,
                        object_name=obj_name,
                        location=info.get('location', 'unknown')
                    )
                    print(f"[Exploration] Swept {obj_name}.")
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj_name}:", e)
                try:
                    print(f"[Exploration] Attempting to rotate {obj_name} at {info.get('location', 'unknown')}")
                    obs, reward, done = execute_rotate(
                        env,
                        task,
                        object_name=obj_name,
                        location=info.get('location', 'unknown')
                    )
                    print(f"[Exploration] Rotated {obj_name}.")
                except Exception as e:
                    print(f"[Exploration] Failed to rotate {obj_name}:", e)

        # Step 7: Try gripper action
        try:
            print("[Exploration] Attempting gripper action")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            print("[Exploration] Gripper action executed.")
        except Exception as e:
            print("[Exploration] Failed to execute gripper action:", e)

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()