# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that (not (need-ready)) is a missing predicate in preconditions.
        # Some skills (e.g., execute_pick_object, execute_place_to_drawer, execute_go, etc.) require (not (need-ready)) to be true.
        # After executing a skill that sets (need-ready), we must call execute_go_ready to clear it.

        # For demonstration, let's assume a simple plan:
        # 1. Move to an object on the floor
        # 2. Pick the object (sets need-ready)
        # 3. Go to ready pose (clears need-ready)
        # 4. Move to a drawer
        # 5. Place the object in the drawer (sets need-ready)
        # 6. Go to ready pose (clears need-ready)
        # (You can expand this plan as needed for your specific task.)

        # For this example, we will use the first object and first drawer found in positions
        # (You should adapt this logic to your actual environment and plan.)

        # Find an object on the floor and a drawer
        object_on_floor = None
        object_on_floor_pos = None
        drawer = None
        drawer_pos = None
        for name, pos in positions.items():
            if 'object' in name and 'on_floor' in name:
                object_on_floor = name
                object_on_floor_pos = pos
            if 'drawer' in name:
                drawer = name
                drawer_pos = pos
            if object_on_floor and drawer:
                break

        if object_on_floor is None or drawer is None:
            print("[Error] Could not find required object or drawer in positions.")
            return

        # Assume robot starts at 'ready-pose'
        robot_location = 'ready-pose'
        object_location = object_on_floor_pos  # This should be a location name or coordinates
        drawer_location = drawer_pos

        # 1. Move to object location
        try:
            print(f"[Task] Moving from {robot_location} to {object_location} to approach {object_on_floor}")
            obs, reward, done = execute_go(env, task, robot_location, object_location)
            robot_location = object_location
        except Exception as e:
            print(f"[Error] Failed to move to object: {e}")
            return

        # 2. Pick the object (sets need-ready)
        try:
            print(f"[Task] Picking up {object_on_floor} at {object_location}")
            obs, reward, done = execute_pick(env, task, object_on_floor, object_location)
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # 3. Go to ready pose (clears need-ready)
        try:
            print(f"[Task] Going to ready-pose to clear need-ready")
            obs, reward, done = execute_go_ready(env, task, object_location)
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready-pose: {e}")
            return

        # 4. Move to drawer location
        try:
            print(f"[Task] Moving from {robot_location} to {drawer_location} to approach {drawer}")
            obs, reward, done = execute_go(env, task, robot_location, drawer_location)
            robot_location = drawer_location
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")
            return

        # 5. Place the object in the drawer (sets need-ready)
        try:
            print(f"[Task] Placing {object_on_floor} into {drawer} at {drawer_location}")
            obs, reward, done = execute_place(env, task, object_on_floor, drawer, drawer_location)
        except Exception as e:
            print(f"[Error] Failed to place object in drawer: {e}")
            return

        # 6. Go to ready pose (clears need-ready)
        try:
            print(f"[Task] Going to ready-pose to clear need-ready after placing")
            obs, reward, done = execute_go_ready(env, task, drawer_location)
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready-pose after placing: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()

# Notes:
# - This code demonstrates the exploration and handling of the (need-ready) predicate.
# - After any skill that sets (need-ready), you must call execute_go_ready to clear it before further actions.
# - Only predefined skills are used, and all control flow is handled in the main function.
# - Error handling is included for each step.
# - You should adapt the object/drawer selection and plan steps to your specific environment and oracle plan.