# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # The only action that clears (need-ready) is execute_go_ready.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it.

        # === Example Plan Execution ===
        # For demonstration, we will:
        # 1. Pick up an object from the floor (execute_pick_object)
        # 2. Go to ready pose (execute_go_ready) to clear (need-ready)
        # 3. Move to a drawer (execute_go)
        # 4. Place the object into the drawer (execute_place_to_drawer)
        # 5. Go to ready pose (execute_go_ready) again if needed

        # For this example, we will use the first object and first location found in positions
        # You should adapt this logic to your actual oracle plan and object names

        # Find an object on the floor and a location
        object_name = None
        location_name = None
        for name, pos in positions.items():
            if 'object' in name or 'obj' in name:
                object_name = name
                break
        for name, pos in positions.items():
            if 'location' in name or 'room' in name or 'pose' in name:
                location_name = name
                break
        # Fallbacks if not found
        if object_name is None:
            object_name = list(positions.keys())[0]
        if location_name is None:
            location_name = list(positions.keys())[0]

        # For demonstration, assume a drawer and a handle exist
        drawer_name = None
        handle_name = None
        for name in positions.keys():
            if 'drawer' in name:
                drawer_name = name
            if 'handle' in name:
                handle_name = name

        # Step 1: Pick up an object from the floor
        try:
            print(f"[Task] Picking up {object_name} at {location_name}")
            obs, reward, done = execute_pick_object(
                env,
                task,
                obj=object_name,
                location=location_name
            )
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # Step 2: Go to ready pose to clear (need-ready)
        try:
            print(f"[Task] Going to ready pose from {location_name} to clear (need-ready)")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=location_name
            )
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # Step 3: Move to drawer location (if different)
        if drawer_name is not None and drawer_name != location_name:
            try:
                print(f"[Task] Moving from {location_name} to {drawer_name}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=location_name,
                    to_location=drawer_name
                )
                current_location = drawer_name
            except Exception as e:
                print(f"[Error] Failed to move to drawer: {e}")
                return
        else:
            current_location = location_name

        # Step 4: Place object into drawer
        if drawer_name is not None:
            try:
                print(f"[Task] Placing {object_name} into {drawer_name} at {current_location}")
                obs, reward, done = execute_place_to_drawer(
                    env,
                    task,
                    obj=object_name,
                    drawer=drawer_name,
                    location=current_location
                )
            except Exception as e:
                print(f"[Error] Failed to place object into drawer: {e}")
                return

            # Step 5: Go to ready pose again to clear (need-ready)
            try:
                print(f"[Task] Going to ready pose from {current_location} to clear (need-ready)")
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=current_location
                )
            except Exception as e:
                print(f"[Error] Failed to go to ready pose: {e}")
                return

        # === Exploration for Missing Predicate ===
        # If at any point an action fails due to (need-ready) being true, the above logic ensures
        # that execute_go_ready is called to clear it. This is the missing predicate handling.

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
