# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicate (need-ready).'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Detect Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # We will attempt to trigger actions and observe if (need-ready) is required for further actions.
        # This phase will try to execute a sequence and check if the environment blocks on (need-ready).

        # For demonstration, we will attempt to pick and place an object, and if blocked, call execute_go_ready.

        # Find a pickable object on the floor (not a handle)
        pickable_obj = None
        pickable_pos = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('on_floor', False) and not obj_info.get('is_handle', False):
                pickable_obj = obj_name
                pickable_pos = obj_info['position']
                break

        if pickable_obj is None:
            print("[Exploration] No pickable object found on the floor.")
        else:
            print(f"[Exploration] Found pickable object: {pickable_obj} at {pickable_pos}")

            # Try to pick the object
            try:
                print(f"[Exploration] Attempting to pick {pickable_obj}...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=pickable_obj,
                    target_pos=pickable_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked {pickable_obj}.")
            except Exception as e:
                print(f"[Exploration] Exception during pick: {e}")

            # Try to place the object into a bin or drawer if available
            place_success = False
            for obj_name, obj_info in positions.items():
                if obj_info.get('is_drawer', False) and obj_info.get('drawer_open', False):
                    drawer_name = obj_name
                    drawer_pos = obj_info['position']
                    try:
                        print(f"[Exploration] Attempting to place {pickable_obj} into {drawer_name}...")
                        obs, reward, done = execute_place(
                            env,
                            task,
                            obj_name=pickable_obj,
                            target_pos=drawer_pos,
                            max_steps=100,
                            threshold=0.01,
                            timeout=10.0
                        )
                        print(f"[Exploration] Placed {pickable_obj} into {drawer_name}.")
                        place_success = True
                        break
                    except Exception as e:
                        print(f"[Exploration] Exception during place: {e}")
            if not place_success:
                # Try to place into bin if drawer not available
                if 'bin' in positions:
                    bin_pos = positions['bin']['position']
                    try:
                        print(f"[Exploration] Attempting to place {pickable_obj} into bin...")
                        obs, reward, done = execute_place(
                            env,
                            task,
                            obj_name=pickable_obj,
                            target_pos=bin_pos,
                            max_steps=100,
                            threshold=0.01,
                            timeout=10.0
                        )
                        print(f"[Exploration] Placed {pickable_obj} into bin.")
                        place_success = True
                    except Exception as e:
                        print(f"[Exploration] Exception during place to bin: {e}")

            # After pick and place, try to perform another pick to see if (need-ready) is required
            try:
                print(f"[Exploration] Attempting to pick {pickable_obj} again to check for (need-ready)...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=pickable_obj,
                    target_pos=pickable_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked {pickable_obj} again.")
            except Exception as e:
                print(f"[Exploration] Exception during second pick: {e}")
                print("[Exploration] Detected possible missing (need-ready) predicate. Attempting to call execute_go_ready...")

                # Try to call execute_go_ready to clear (need-ready)
                try:
                    # Assume robot's current location is available as 'robot' in positions
                    robot_pos = None
                    for obj_name, obj_info in positions.items():
                        if obj_name == 'robot':
                            robot_pos = obj_info['position']
                            break
                    if robot_pos is None:
                        # Fallback: use pickable_pos as current location
                        robot_pos = pickable_pos
                    obs, reward, done = execute_go_ready(
                        env,
                        task,
                        from_pos=robot_pos,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print("[Exploration] Called execute_go_ready to clear (need-ready).")
                except Exception as e2:
                    print(f"[Exploration] Exception during execute_go_ready: {e2}")

                # Try the pick again after go_ready
                try:
                    print(f"[Exploration] Retrying pick {pickable_obj} after go_ready...")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj_name=pickable_obj,
                        target_pos=pickable_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Picked {pickable_obj} after go_ready.")
                except Exception as e3:
                    print(f"[Exploration] Still unable to pick after go_ready: {e3}")

        # === End of Exploration Phase ===

        print("[Exploration] Exploration phase complete. If (need-ready) was required, it should now be handled by calling execute_go_ready between actions as needed.")

        # === Main Task Plan (Oracle Plan Execution) ===
        # At this point, you would execute the actual oracle plan, inserting execute_go_ready
        # between actions if (need-ready) is set after certain actions (e.g., after pick/place).
        # For each action in the plan:
        #   1. Execute the action (e.g., execute_pick, execute_place, etc.)
        #   2. If the action sets (need-ready), call execute_go_ready before the next action.

        # Example (pseudo-code, replace with actual plan steps):
        # for step in oracle_plan:
        #     if step['action'] == 'execute_pick':
        #         execute_pick(...)
        #         execute_go_ready(...)
        #     elif step['action'] == 'execute_place':
        #         execute_place(...)
        #         execute_go_ready(...)
        #     elif step['action'] == 'execute_push':
        #         execute_push(...)
        #     elif step['action'] == 'execute_pull':
        #         execute_pull(...)
        #     elif step['action'] == 'execute_go':
        #         execute_go(...)
        #     # etc.

        # TODO: Insert actual oracle plan execution here, using only predefined skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()