# run_skeleton_task.py (Completed Executable with Exploration for (need-ready))


import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *   # Use ONLY provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # === INITIAL STATE GETTING ===
        # We try to infer what "need-ready" is missing in the state
        # Most actions in the provided domain (see :precondition (not (need-ready)))
        # So, after each relevant skill we should check the signal or trigger returning to ready pose
        
        #--- EXPLORATION PHASE: Try available skills, look for missing predicate issues ---

        # We will attempt to execute each available skill and see if the environment
        # blocks us at certain states (e.g., because (need-ready) is present unexpectedly).
        # If so, we conclude (need-ready) is a blocking predicate and must be cleared by returning to ready.

        skill_sequence = [
            ('execute_pick', {}),
            ('execute_place', {}),
            ('execute_push', {}),
            ('execute_pull', {}),
            ('execute_sweep', {}),
            ('execute_rotate', {}),
            ('execute_go', {}),
            ('execute_gripper', {}),
        ]
        
        # For exploration, try to call each skill with dummy/default arguments if possible and
        # capture exceptions or failed preconditions due to (need-ready).
        # Since dummy arguments may not exist, we wrap in try/except and print any clues.
        for skill_name, kwargs in skill_sequence:
            skill_fn = globals().get(skill_name)
            if skill_fn is None:
                continue
            try:
                # Attempt with dummy/fallback arguments based on known object/position keys
                # (In practice these would be determined from the observation/init state).
                # Here we just grab arbitrary objects/locations if present.
                
                arglist = []
                if skill_name == 'execute_pick':  # Try to pick an object from the floor
                    # Find any object on the floor
                    floor_objs = [k for k, v in positions.items() if 'floor' in k or 'obj' in k]
                    if len(floor_objs) > 0:
                        obj = floor_objs[0]
                        # Try to find a plausible place
                        for posname in positions:
                            if 'room' in posname or 'loc' in posname:
                                location = posname
                                break
                        else:
                            location = 'table'
                        arglist = [env, task, obj, location]
                elif skill_name == 'execute_place':
                    # Place a held object into somewhere
                    # No-op here - depends on prior pick
                    arglist = [env, task, 'object', 'drawer', 'location']  # placeholders
                elif skill_name == 'execute_push':
                    arglist = [env, task, 'drawer', 'location']
                elif skill_name == 'execute_pull':
                    arglist = [env, task, 'drawer', 'handle', 'location']
                elif skill_name == 'execute_sweep':
                    arglist = [env, task, 'object', 'location']
                elif skill_name == 'execute_rotate':
                    arglist = [env, task, 'object', 'location']
                elif skill_name == 'execute_go':
                    arglist = [env, task, 'location1', 'location2']
                elif skill_name == 'execute_gripper':
                    arglist = [env, task]

                if len(arglist) > 0:
                    print(f"[Exploration] Trying skill: {skill_name} with args: {arglist}")
                    obs, reward, done = skill_fn(*arglist)
                    print(f"[Exploration] Skill {skill_name} returned: done={done}")
            except Exception as e:
                # Print exception -- we are looking for clues that (need-ready) precludes actions
                print(f"[Exploration] Skill {skill_name} failed with exception: {e}")
                # If error indicates (need-ready) is blocking, we have confirmed the missing predicate

        print("[Exploration] Completed. If skills failed at (need-ready), you must first return to ready pose to clear (need-ready).")
        print("-------------")

        # === PLAN EXECUTION PHASE ===
        # Now, to properly execute the plan, after every action with :effect (need-ready), 
        # and before the next action that requires (not (need-ready)), 
        # you must invoke the 'execute_go_ready' skill (if provided), or whatever skill returns robot to ready pose.

        # We illustrate an example "oracle" plan, with (need-ready) handling:

        # 1. Pick an object from the floor
        # 2. (need-ready) now True, so must go to ready
        # 3. Only now continue with next action

        # For illustration, let's process one object if such exists:
        try:
            # Find any object lying on the floor
            floor_objects = [k for k, v in positions.items() if 'floor' in k or 'obj' in k]
            if len(floor_objects) > 0:
                obj = floor_objects[0]
                # Find any plausible location
                location = None
                for loc in positions:
                    if 'table' in loc or 'room' in loc or 'floor' in loc or 'bin' in loc or 'drawer' in loc:
                        location = loc
                        break
                if location is None:
                    location = list(positions.keys())[0]
                print(f"[Task] Picking object {obj} at {location}")
                obs, reward, done = execute_pick(env, task, obj, location)
                print(f"[Task] Picked. Now (need-ready)=True (requiring robot to return to ready-pose)")
                
                # After picking, according to domain, (need-ready) is now True.
                # Must return to ready pose (invoke execute_go_ready)
                # Only then is (need-ready) cleared
                
                print("[Task] Returning to ready pose with execute_go_ready")
                obs, reward, done = execute_go_ready(env, task, location)
                print("[Task] Returned to ready pose; (need-ready) cleared")

                # Now continue with further actions if desired
                # For demonstration, place the object (if 'execute_place' exists and makes sense)
                # Here, we can continue further steps (e.g. execute_place, execute_go, etc.) as needed
        except Exception as e:
            print(f"[Task] Exception while performing main plan: {e}")

        # You may continue/expand this plan according to your specific problem template
        # The KEY is: after any skill whose effect sets (need-ready), you must call execute_go_ready (with current location)
        # before executing the next skill needing (not (need-ready)) in its precondition.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
