# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate (drawer-unlocked drawer_top) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer_top) is missing.
        # We need to explore the environment to determine the lock state of the drawer.
        # We will use the available skills to do this.

        # Assumptions:
        # - There is a drawer named 'drawer_top' in the environment.
        # - The robot starts at some initial location.
        # - The handle for the drawer is on the floor and must be picked up.
        # - The robot must move to the drawer, pick up the handle, and attempt to pull the drawer.
        # - If the drawer is locked, the pull will fail or indicate the lock state.

        # Step 1: Identify relevant objects and locations
        try:
            drawer_name = 'drawer_top'
            handle_name = None
            for obj in positions:
                if 'handle' in obj and drawer_name in obj:
                    handle_name = obj
                    break
            if handle_name is None:
                # Fallback: find any handle
                for obj in positions:
                    if 'handle' in obj:
                        handle_name = obj
                        break
            if handle_name is None:
                raise Exception("No handle found for exploration.")

            # Get locations
            robot_location = None
            for obj in positions:
                if 'robot' in obj:
                    robot_location = positions[obj]
                    break
            if robot_location is None:
                # Fallback: use a default location
                robot_location = (0, 0, 0)

            handle_location = positions.get(handle_name, None)
            drawer_location = positions.get(drawer_name, None)
            if handle_location is None or drawer_location is None:
                raise Exception("Handle or drawer location not found.")

        except Exception as e:
            print(f"[Exploration] Error identifying objects/locations: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move to handle and pick it up
        try:
            print(f"[Exploration] Moving to handle '{handle_name}' at {handle_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',  # Assuming robot starts at ready-pose
                to_location=handle_location
            )
            if done:
                print("[Exploration] Task ended unexpectedly during move to handle.")
                return

            print(f"[Exploration] Picking up handle '{handle_name}'")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=handle_name,
                location=handle_location
            )
            if done:
                print("[Exploration] Task ended unexpectedly during pick handle.")
                return

        except Exception as e:
            print(f"[Exploration] Error during handle pickup: {e}")
            shutdown_environment(env)
            return

        # Step 3: Move to drawer
        try:
            print(f"[Exploration] Moving to drawer '{drawer_name}' at {drawer_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=handle_location,
                to_location=drawer_location
            )
            if done:
                print("[Exploration] Task ended unexpectedly during move to drawer.")
                return
        except Exception as e:
            print(f"[Exploration] Error during move to drawer: {e}")
            shutdown_environment(env)
            return

        # Step 4: Attempt to pull the drawer to check lock state
        try:
            print(f"[Exploration] Attempting to pull drawer '{drawer_name}' with handle '{handle_name}'")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name=handle_name,
                location=drawer_location
            )
            # If the drawer is locked, the skill or environment should indicate this (e.g., via reward, obs, or done)
            # For this exploration, we assume that if the action fails or does not open the drawer, it is locked.
            # The feedback (drawer-unlocked drawer_top) is what we are trying to discover.
            # Here, you would check the observation or environment state to see if the drawer is now open or still closed.
            # For demonstration, we print the result.
            if done:
                print("[Exploration] Task ended after pull attempt.")
                return
            else:
                print("[Exploration] Pull attempt complete. Check environment state for lock status.")

        except Exception as e:
            print(f"[Exploration] Error during pull attempt: {e}")
            shutdown_environment(env)
            return

        # Step 5: (Optional) Record the discovered predicate
        # In a real system, you would now update your knowledge base with (drawer-unlocked drawer_top)
        print("[Exploration] Exploration phase complete. Predicate (drawer-unlocked drawer_top) can now be set.")

        # === End of Exploration Phase ===

        # You can now proceed with the main oracle plan, using the discovered predicate as needed.
        # For this task, the focus was on exploration and predicate discovery.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
