# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # The only way to clear (need-ready) is to execute the 'execute_go_ready' action.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready before the next action.

        # === Example Plan Execution ===
        # For demonstration, we will simulate a generic plan that:
        # 1. Picks up an object from the floor (execute_pick)
        # 2. Moves to a drawer (execute_go)
        # 3. Places the object into the drawer (execute_place)
        # 4. Pushes the drawer closed (execute_push)
        # 5. Returns to ready pose if needed

        # NOTE: Replace 'object_1', 'drawer_1', 'location_1', etc. with actual names from your environment.
        # For this example, we will use placeholder names and handle missing keys gracefully.

        # Helper function to get a position or return None
        def get_pos_safe(name):
            return positions[name] if name in positions else None

        # Example object and location names (replace with actual names as needed)
        object_name = None
        drawer_name = None
        handle_name = None
        location_pick = None
        location_drawer = None
        location_bin = None

        # Try to infer object and location names from positions
        for key in positions:
            if 'object' in key and object_name is None:
                object_name = key
            if 'drawer' in key and drawer_name is None:
                drawer_name = key
            if 'handle' in key and handle_name is None:
                handle_name = key
            if 'floor' in key and location_pick is None:
                location_pick = key
            if 'bin' in key and location_bin is None:
                location_bin = key
            if 'drawer' in key and location_drawer is None:
                location_drawer = key

        # Fallbacks if not found
        if object_name is None and len(positions) > 0:
            object_name = list(positions.keys())[0]
        if location_pick is None and len(positions) > 0:
            location_pick = list(positions.keys())[0]
        if location_drawer is None and len(positions) > 1:
            location_drawer = list(positions.keys())[1]

        # Track if need-ready is set
        need_ready = False

        # 1. Pick up object from floor
        try:
            print(f"[Task] Attempting to pick up {object_name} at {location_pick}")
            obs, reward, done = execute_pick(env, task, object_name, location_pick)
            need_ready = True
        except Exception as e:
            print(f"[Error] Failed to pick up object: {e}")

        # If need-ready is set, go to ready pose
        if need_ready:
            try:
                print("[Task] Executing go_ready to clear need-ready")
                obs, reward, done = execute_go_ready(env, task, location_pick)
                need_ready = False
            except Exception as e:
                print(f"[Error] Failed to execute go_ready: {e}")

        # 2. Move to drawer location
        try:
            print(f"[Task] Moving to drawer location: {location_drawer}")
            obs, reward, done = execute_go(env, task, location_pick, location_drawer)
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")

        # 3. Place object into drawer
        try:
            print(f"[Task] Placing {object_name} into {drawer_name} at {location_drawer}")
            obs, reward, done = execute_place(env, task, object_name, drawer_name, location_drawer)
            need_ready = True
        except Exception as e:
            print(f"[Error] Failed to place object into drawer: {e}")

        # If need-ready is set, go to ready pose
        if need_ready:
            try:
                print("[Task] Executing go_ready to clear need-ready")
                obs, reward, done = execute_go_ready(env, task, location_drawer)
                need_ready = False
            except Exception as e:
                print(f"[Error] Failed to execute go_ready: {e}")

        # 4. Push drawer closed
        try:
            print(f"[Task] Pushing drawer {drawer_name} closed at {location_drawer}")
            obs, reward, done = execute_push(env, task, drawer_name, location_drawer)
        except Exception as e:
            print(f"[Error] Failed to push drawer closed: {e}")

        # 5. (Optional) Place object into bin (if required by your plan)
        # try:
        #     print(f"[Task] Placing {object_name} into bin at {location_bin}")
        #     obs, reward, done = execute_place(env, task, object_name, 'bin', location_bin)
        #     need_ready = True
        # except Exception as e:
        #     print(f"[Error] Failed to place object into bin: {e}")

        # if need_ready:
        #     try:
        #         print("[Task] Executing go_ready to clear need-ready")
        #         obs, reward, done = execute_go_ready(env, task, location_bin)
        #         need_ready = False
        #     except Exception as e:
        #         print(f"[Error] Failed to execute go_ready: {e}")

        # === Exploration: Check for missing predicates ===
        # The feedback and exploration domain suggest that (need-ready) is a critical predicate.
        # If actions fail due to (need-ready), ensure execute_go_ready is called before the next action.

        print("[Task] Skeleton task completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
