# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it.

        # For demonstration, let's assume a simple plan:
        # 1. Move to an object on the floor
        # 2. Pick the object (sets need-ready)
        # 3. Go to ready pose (clears need-ready)
        # 4. Move to a drawer
        # 5. Place the object in the drawer (sets need-ready)
        # 6. Go to ready pose (clears need-ready)
        # 7. Done

        # For this example, we will use the first object and first drawer found in positions
        # (You may need to adapt this logic to your actual environment and object naming)

        # Find an object on the floor and a drawer
        object_name = None
        object_pos = None
        drawer_name = None
        drawer_pos = None
        for name, pos in positions.items():
            if 'object' in name or 'obj' in name:
                object_name = name
                object_pos = pos
                break
        for name, pos in positions.items():
            if 'drawer' in name:
                drawer_name = name
                drawer_pos = pos
                break

        if object_name is None or drawer_name is None:
            print("[Error] Could not find required object or drawer in positions.")
            return

        # Assume location names are available as keys in positions
        # For demonstration, we use the keys directly
        # You may need to adapt this to your environment's naming conventions

        # Step 1: Move to the object's location
        try:
            print(f"[Task] Moving to object {object_name} at {object_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',  # Assuming robot starts at ready-pose
                to_location=object_name,      # Use object name as location key
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to move to object: {e}")
            return

        # Step 2: Pick the object (sets need-ready)
        try:
            print(f"[Task] Picking object {object_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_name,
                location=object_name,  # Use object name as location key
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to pick object: {e}")
            return

        # Step 3: Go to ready pose (clears need-ready)
        try:
            print(f"[Task] Going to ready pose to clear need-ready")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=object_name,  # From where we picked
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # Step 4: Move to drawer location
        try:
            print(f"[Task] Moving to drawer {drawer_name} at {drawer_pos}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location=drawer_name,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")
            return

        # Step 5: Place the object in the drawer (sets need-ready)
        try:
            print(f"[Task] Placing object {object_name} into drawer {drawer_name}")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_name,
                drawer_name=drawer_name,
                location=drawer_name,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to place object in drawer: {e}")
            return

        # Step 6: Go to ready pose (clears need-ready)
        try:
            print(f"[Task] Going to ready pose to clear need-ready after placing")
            obs, reward, done = execute_go_ready(
                env,
                task,
                from_location=drawer_name,
                max_steps=100
            )
        except Exception as e:
            print(f"[Error] Failed to go to ready pose after placing: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
