# run_skeleton_task.py (Completed with Exploration for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # The only action that clears (need-ready) is execute_go_ready.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it.

        # === Example Plan Execution ===
        # For demonstration, we will:
        # 1. Pick up an object from the floor (execute_pick)
        # 2. Go to ready pose (execute_go_ready) to clear (need-ready)
        # 3. Place the object into a drawer (execute_place)
        # 4. Go to ready pose (execute_go_ready) again if needed

        # For this example, we will use the first object and first location found in positions
        # You should adapt this logic to your actual oracle plan and object names

        # Find an object on the floor and a drawer
        object_on_floor = None
        object_pos = None
        drawer = None
        drawer_pos = None
        for name, pos in positions.items():
            if 'drawer' in name and drawer is None:
                drawer = name
                drawer_pos = pos
            elif 'object' in name and object_on_floor is None:
                object_on_floor = name
                object_pos = pos

        if object_on_floor is None or drawer is None:
            print("[Error] Could not find required objects in the environment.")
            return

        # Assume the robot starts at some initial location
        robot_location = 'init_pos'
        if 'robot' in positions:
            robot_location = positions['robot']

        # 1. Move to the object's location if not already there
        try:
            if robot_location != object_pos:
                print(f"[Task] Moving to object {object_on_floor} at {object_pos}")
                obs, reward, done = execute_go(env, task, robot_location, object_pos)
                robot_location = object_pos
        except Exception as e:
            print(f"[Error] Failed to move to object: {e}")
            return

        # 2. Pick up the object from the floor
        try:
            print(f"[Task] Picking up object {object_on_floor} at {object_pos}")
            obs, reward, done = execute_pick(env, task, object_on_floor, object_pos)
            # After this, (need-ready) is set, so we must call execute_go_ready
        except Exception as e:
            print(f"[Error] Failed to pick up object: {e}")
            return

        # 3. Go to ready pose to clear (need-ready)
        try:
            print("[Task] Going to ready pose to clear (need-ready)")
            obs, reward, done = execute_go_ready(env, task, object_pos)
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose: {e}")
            return

        # 4. Move to the drawer location
        try:
            if robot_location != drawer_pos:
                print(f"[Task] Moving to drawer {drawer} at {drawer_pos}")
                obs, reward, done = execute_go(env, task, robot_location, drawer_pos)
                robot_location = drawer_pos
        except Exception as e:
            print(f"[Error] Failed to move to drawer: {e}")
            return

        # 5. Place the object into the drawer
        try:
            print(f"[Task] Placing object {object_on_floor} into drawer {drawer} at {drawer_pos}")
            obs, reward, done = execute_place(env, task, object_on_floor, drawer, drawer_pos)
            # After this, (need-ready) is set again, so we must call execute_go_ready
        except Exception as e:
            print(f"[Error] Failed to place object: {e}")
            return

        # 6. Go to ready pose to clear (need-ready) again
        try:
            print("[Task] Going to ready pose to clear (need-ready) after placing")
            obs, reward, done = execute_go_ready(env, task, drawer_pos)
            robot_location = 'ready-pose'
        except Exception as e:
            print(f"[Error] Failed to go to ready pose after placing: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()