# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # According to the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # The only action that clears (need-ready) is execute_go_ready.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it before the next action.

        # For demonstration, we will perform a minimal exploration:
        # 1. Try to pick an object (using execute_pick or execute_pick_object).
        # 2. If the action fails due to (need-ready), call execute_go_ready and retry.
        # 3. This will confirm the missing predicate and the need for the ready-pose step.

        # For this example, we will attempt to pick the first object found on the floor.
        # The actual object names and positions depend on the environment; we use the positions dict.

        # Find an object on the floor (excluding handles)
        object_to_pick = None
        object_location = None
        for obj_name, obj_info in positions.items():
            # obj_info is expected to be a dict with keys like 'type', 'location', 'on_floor', etc.
            if obj_info.get('on_floor', False) and not obj_info.get('is_handle', False):
                object_to_pick = obj_name
                object_location = obj_info.get('location')
                break

        if object_to_pick is None or object_location is None:
            print("[Exploration] No suitable object found on the floor to pick.")
        else:
            print(f"[Exploration] Attempting to pick object '{object_to_pick}' at location '{object_location}'.")

            # Try to pick the object
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=object_to_pick,
                    location=object_location
                )
                print(f"[Exploration] Picked object '{object_to_pick}' successfully.")
            except Exception as e:
                # If the error is related to (need-ready), handle it
                print(f"[Exploration] Exception during pick: {e}")
                print("[Exploration] Attempting to clear (need-ready) by calling execute_go_ready.")

                # Try to move to ready pose to clear (need-ready)
                try:
                    obs, reward, done = execute_go_ready(
                        env,
                        task,
                        from_location=object_location
                    )
                    print("[Exploration] Cleared (need-ready) by moving to ready pose.")
                except Exception as e2:
                    print(f"[Exploration] Exception during execute_go_ready: {e2}")

                # Retry picking the object
                try:
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj=object_to_pick,
                        location=object_location
                    )
                    print(f"[Exploration] Picked object '{object_to_pick}' successfully after clearing (need-ready).")
                except Exception as e3:
                    print(f"[Exploration] Failed to pick object after clearing (need-ready): {e3}")

        # === End of Exploration Phase ===

        # TODO: Continue with further task logic or exploration as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
