# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # According to the domain, after certain actions (e.g., pick, place), (need-ready) becomes true,
        # and the robot must execute 'execute_go_ready' to return to ready-pose and clear (need-ready).
        # If (need-ready) is true, no other action except 'execute_go_ready' can be executed.

        # We will simulate a minimal exploration to trigger (need-ready) and then resolve it.

        # For demonstration, let's:
        # 1. Pick an object from the floor (triggers (need-ready))
        # 2. Try to pick another object (should fail if (need-ready) is not cleared)
        # 3. Execute 'execute_go_ready' to clear (need-ready)
        # 4. Pick another object (should now succeed)

        # Find two objects on the floor and a location for the robot
        floor_objects = []
        robot_location = None
        for name, info in positions.items():
            if info.get('on_floor', False):
                floor_objects.append(name)
            if info.get('is_robot', False):
                robot_location = info.get('location', None)
        if len(floor_objects) < 2 or robot_location is None:
            print("[Exploration] Not enough objects or missing robot location for exploration.")
            return

        obj1 = floor_objects[0]
        obj2 = floor_objects[1]
        loc = robot_location

        print(f"[Exploration] Attempting to pick {obj1} at {loc} (should set need-ready).")
        try:
            obs, reward, done = execute_pick(env, task, obj1, loc)
            print(f"[Exploration] Picked {obj1}. (need-ready) should now be True.")
        except Exception as e:
            print(f"[Exploration] Failed to pick {obj1}: {e}")
            return

        print(f"[Exploration] Attempting to pick {obj2} at {loc} (should fail if need-ready is set).")
        try:
            obs, reward, done = execute_pick(env, task, obj2, loc)
            print(f"[Exploration] Unexpectedly picked {obj2} (need-ready may not be enforced).")
        except Exception as e:
            print(f"[Exploration] As expected, failed to pick {obj2} due to (need-ready): {e}")

        print(f"[Exploration] Executing go_ready to clear (need-ready).")
        try:
            obs, reward, done = execute_go_ready(env, task, loc)
            print(f"[Exploration] Executed go_ready. (need-ready) should now be False.")
        except Exception as e:
            print(f"[Exploration] Failed to execute go_ready: {e}")
            return

        print(f"[Exploration] Attempting to pick {obj2} at {loc} again (should now succeed).")
        try:
            obs, reward, done = execute_pick(env, task, obj2, loc)
            print(f"[Exploration] Successfully picked {obj2} after clearing (need-ready).")
        except Exception as e:
            print(f"[Exploration] Still failed to pick {obj2}: {e}")

        print("[Exploration] Exploration phase complete. (need-ready) predicate is required and must be handled after certain actions.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
