# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # The only action that clears (need-ready) is execute_go_ready.
        # Therefore, after any action that sets (need-ready), we must call execute_go_ready to clear it.

        # For demonstration, we will perform a minimal exploration:
        # 1. Try to pick an object (which sets need-ready)
        # 2. Try to pick another object without clearing need-ready (should fail)
        # 3. Call execute_go_ready to clear need-ready
        # 4. Try to pick again (should succeed)

        # For this, we need at least one object on the floor and the robot's location.
        # We'll use the first object found on the floor.

        # Find an object on the floor and its location
        object_on_floor = None
        object_pos = None
        robot_location = None

        # Try to extract a suitable object and robot location from positions
        for obj_name, pos_info in positions.items():
            if isinstance(pos_info, dict) and pos_info.get('on_floor', False):
                object_on_floor = obj_name
                object_pos = pos_info.get('location', None)
                break

        # Fallback: just pick any object and location if not found
        if object_on_floor is None:
            for obj_name, pos_info in positions.items():
                if isinstance(pos_info, dict) and 'location' in pos_info:
                    object_on_floor = obj_name
                    object_pos = pos_info['location']
                    break

        # Get robot's current location
        for key, pos_info in positions.items():
            if key.startswith('robot') and isinstance(pos_info, dict) and 'location' in pos_info:
                robot_location = pos_info['location']
                break

        # If not found, set a default
        if robot_location is None:
            robot_location = 'ready-pose'

        # If no object found, abort exploration
        if object_on_floor is None or object_pos is None:
            print("[Exploration] No suitable object found on the floor for exploration.")
        else:
            print(f"[Exploration] Using object '{object_on_floor}' at location '{object_pos}'.")

            # 1. Move robot to the object's location if not already there
            if robot_location != object_pos:
                try:
                    print(f"[Exploration] Moving robot from '{robot_location}' to '{object_pos}'.")
                    obs, reward, done = execute_go(env, task, robot_location, object_pos)
                    robot_location = object_pos
                except Exception as e:
                    print(f"[Exploration] Error during execute_go: {e}")

            # 2. Try to pick the object (should succeed, sets need-ready)
            try:
                print(f"[Exploration] Attempting to pick '{object_on_floor}' at '{object_pos}'.")
                obs, reward, done = execute_pick(env, task, object_on_floor, object_pos)
                print("[Exploration] Picked object. (need-ready) should now be set.")
            except Exception as e:
                print(f"[Exploration] Error during execute_pick: {e}")

            # 3. Try to pick again without clearing need-ready (should fail)
            try:
                print(f"[Exploration] Attempting to pick '{object_on_floor}' again without clearing need-ready.")
                obs, reward, done = execute_pick(env, task, object_on_floor, object_pos)
                print("[Exploration] Unexpectedly succeeded in picking again (should not happen).")
            except Exception as e:
                print(f"[Exploration] As expected, failed to pick again due to (need-ready): {e}")

            # 4. Call execute_go_ready to clear need-ready
            try:
                print(f"[Exploration] Calling execute_go_ready to clear (need-ready).")
                obs, reward, done = execute_go_ready(env, task, object_pos)
                print("[Exploration] (need-ready) cleared.")
                robot_location = 'ready-pose'
            except Exception as e:
                print(f"[Exploration] Error during execute_go_ready: {e}")

            # 5. Try to pick again (should now succeed)
            try:
                print(f"[Exploration] Attempting to pick '{object_on_floor}' again after clearing need-ready.")
                obs, reward, done = execute_pick(env, task, object_on_floor, object_pos)
                print("[Exploration] Successfully picked object after clearing (need-ready).")
            except Exception as e:
                print(f"[Exploration] Error during execute_pick after clearing need-ready: {e}")

        # === End of Exploration Phase ===

        # TODO: Insert oracle plan execution here, using the same pattern:
        # After any action that sets (need-ready), call execute_go_ready before the next action that requires (not (need-ready)).

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
