# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the environment to determine the lock state of drawer1.
        # We'll use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Assumptions for object names and locations (update as per your environment)
        robot_location = None
        drawer_name = 'drawer1'
        handle_name = None
        handle_location = None
        drawer_location = None

        # Find handle and drawer locations from positions
        for obj_name, pos in positions.items():
            if 'handle' in obj_name:
                handle_name = obj_name
                handle_location = pos
            if drawer_name in obj_name:
                drawer_location = pos

        # Fallback: If drawer location is not found, use a default or raise error
        if drawer_location is None:
            print("[Exploration] Drawer location not found in positions. Please check object_positions.")
            return

        # Fallback: If handle is not found, skip handle-related actions
        if handle_name is None or handle_location is None:
            print("[Exploration] Handle not found in positions. Please check object_positions.")
            return

        # Get robot's initial location (assume from observation or positions)
        for obj_name, pos in positions.items():
            if 'robot' in obj_name or 'gripper' in obj_name:
                robot_location = pos
                break
        if robot_location is None:
            # Try to get from observation
            if hasattr(task, 'get_robot_position'):
                robot_location = task.get_robot_position()
            else:
                print("[Exploration] Robot location not found. Please check environment setup.")
                return

        # 1. Move robot to handle location if not already there
        try:
            if not np.allclose(robot_location, handle_location, atol=0.05):
                print(f"[Exploration] Moving robot to handle at {handle_location}")
                obs, reward, done = execute_go(env, task, from_pos=robot_location, to_pos=handle_location)
                robot_location = handle_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go to handle: {e}")
            return

        # 2. Pick the handle (if not already holding)
        try:
            print(f"[Exploration] Picking up handle: {handle_name}")
            obs, reward, done = execute_pick(env, task, obj_name=handle_name, location=handle_location)
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")
            return

        # 3. Move robot to drawer location (if handle and drawer are not colocated)
        try:
            if not np.allclose(handle_location, drawer_location, atol=0.05):
                print(f"[Exploration] Moving robot to drawer at {drawer_location}")
                obs, reward, done = execute_go(env, task, from_pos=handle_location, to_pos=drawer_location)
                robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Error during execute_go to drawer: {e}")
            return

        # 4. Attempt to pull the drawer using the handle to check lock state
        try:
            print(f"[Exploration] Attempting to pull drawer: {drawer_name} with handle: {handle_name}")
            obs, reward, done = execute_pull(env, task, drawer_name=drawer_name, handle_name=handle_name, location=drawer_location)
            # If the drawer opens, we can infer it was unlocked
            print("[Exploration] Pull action executed. If drawer opened, it is unlocked.")
        except Exception as e:
            print(f"[Exploration] Error during execute_pull: {e}")
            print("[Exploration] Drawer may be locked or another issue occurred.")

        # 5. (Optional) Push the drawer closed again to restore state
        try:
            print(f"[Exploration] Attempting to push drawer: {drawer_name} closed.")
            obs, reward, done = execute_push(env, task, drawer_name=drawer_name, location=drawer_location)
        except Exception as e:
            print(f"[Exploration] Error during execute_push: {e}")

        # 6. Exploration complete. The result of the pull action should reveal the lock state.
        print("[Exploration] Exploration phase complete. Check logs or environment state for (drawer-unlocked drawer1).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()