# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must call the skill that clears it.
        # The action that clears (need-ready) is execute_go_ready.

        # For demonstration, we will perform a minimal exploration:
        # 1. Try to pick an object (if available) using execute_pick or execute_pick_object.
        # 2. If the action sets (need-ready), immediately call execute_go_ready to clear it.
        # 3. Continue with the next action.

        # === Example: Find an object on the floor and pick it up ===
        # We'll look for an object that is on the floor and not a handle.
        # We'll also need the robot's current location.

        # The following is a generic exploration loop to find and pick an object, then handle (need-ready).

        # Get all objects and their positions
        object_names = [name for name in positions.keys() if not name.startswith('robot') and not name.startswith('drawer')]
        robot_pos = None
        for name in positions:
            if name.startswith('robot'):
                robot_pos = positions[name]
                break

        # Find a pickable object (not a handle, on the floor)
        pickable_obj = None
        pickable_obj_pos = None
        for obj in object_names:
            # For this example, we assume the object is on the floor if its z-coordinate is below a threshold
            pos = positions[obj]
            if pos[2] < 0.1:  # z < 0.1 means on the floor (heuristic)
                pickable_obj = obj
                pickable_obj_pos = pos
                break

        if pickable_obj is not None and robot_pos is not None:
            print(f"[Exploration] Attempting to pick object '{pickable_obj}' at position {pickable_obj_pos}")

            # 1. Move to the object's location if not already there
            if np.linalg.norm(np.array(robot_pos) - np.array(pickable_obj_pos)) > 0.05:
                try:
                    print(f"[Exploration] Moving robot to object location: {pickable_obj_pos}")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_pos=robot_pos,
                        to_pos=pickable_obj_pos,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    robot_pos = pickable_obj_pos
                except Exception as e:
                    print(f"[Exploration] Error during execute_go: {e}")

            # 2. Try to pick the object
            try:
                print(f"[Exploration] Picking object: {pickable_obj}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj=pickable_obj,
                    target_pos=pickable_obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                # After picking, (need-ready) is likely set. We must clear it.
                print(f"[Exploration] Executing go_ready to clear (need-ready)")
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_pos=robot_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                # Update robot position to ready-pose if needed
                robot_pos = positions.get('ready-pose', robot_pos)
            except Exception as e:
                print(f"[Exploration] Error during execute_pick or execute_go_ready: {e}")

        else:
            print("[Exploration] No suitable object found to pick.")

        # === End of Exploration Phase ===

        # You can continue with further plan steps here, always checking if (need-ready) is set after actions,
        # and calling execute_go_ready as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()