# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We will use available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and observe the environment's response.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to interact with all objects and drawers to see if any action fails due to a missing predicate
        # This will help us infer which predicate is missing

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('drawer') and not name.startswith('bin')]
        drawer_names = [name for name in positions if name.startswith('drawer')]
        bin_names = [name for name in positions if name.startswith('bin')]
        location_names = list(set([positions[name] for name in positions]))

        # Try to pick each object from the floor
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to pick object: {obj}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=obj,
                    location=positions[obj],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully picked {obj}")
                else:
                    print(f"[Exploration] Could not pick {obj} (possibly missing predicate or blocked)")
            except Exception as e:
                print(f"[Exploration] Exception during pick of {obj}: {e}")

        # Try to pull each drawer using available handles (if any)
        for drawer in drawer_names:
            handle_name = f"{drawer}_handle"
            if handle_name in positions:
                try:
                    print(f"[Exploration] Trying to pull drawer: {drawer} with handle: {handle_name}")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name=drawer,
                        handle_name=handle_name,
                        location=positions[drawer],
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Successfully pulled {drawer}")
                    else:
                        print(f"[Exploration] Could not pull {drawer} (possibly missing predicate or blocked)")
                except Exception as e:
                    print(f"[Exploration] Exception during pull of {drawer}: {e}")

        # Try to place each object into each drawer (if possible)
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Trying to place {obj} into {drawer}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj_name=obj,
                        drawer_name=drawer,
                        location=positions[drawer],
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} into {drawer}")
                    else:
                        print(f"[Exploration] Could not place {obj} into {drawer} (possibly missing predicate or blocked)")
                except Exception as e:
                    print(f"[Exploration] Exception during place of {obj} into {drawer}: {e}")

        # Try to push each drawer closed
        for drawer in drawer_names:
            try:
                print(f"[Exploration] Trying to push drawer: {drawer}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawer,
                    location=positions[drawer],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully pushed {drawer}")
                else:
                    print(f"[Exploration] Could not push {drawer} (possibly missing predicate or blocked)")
            except Exception as e:
                print(f"[Exploration] Exception during push of {drawer}: {e}")

        # Try to sweep each object
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to sweep object: {obj}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj_name=obj,
                    location=positions[obj],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}")
                else:
                    print(f"[Exploration] Could not sweep {obj} (possibly missing predicate or blocked)")
            except Exception as e:
                print(f"[Exploration] Exception during sweep of {obj}: {e}")

        # Try to rotate each object (if applicable)
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to rotate object: {obj}")
                obs, reward, done = execute_rotate(
                    env,
                    task,
                    obj_name=obj,
                    location=positions[obj],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully rotated {obj}")
                else:
                    print(f"[Exploration] Could not rotate {obj} (possibly missing predicate or blocked)")
            except Exception as e:
                print(f"[Exploration] Exception during rotate of {obj}: {e}")

        # Try to move the robot between all pairs of locations
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying to move from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_location=from_loc,
                            to_location=to_loc,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                        else:
                            print(f"[Exploration] Could not move from {from_loc} to {to_loc} (possibly missing predicate or blocked)")
                    except Exception as e:
                        print(f"[Exploration] Exception during move from {from_loc} to {to_loc}: {e}")

        # Try to use the gripper (no-op, but included for completeness)
        try:
            print(f"[Exploration] Trying to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print(f"[Exploration] Successfully used gripper")
            else:
                print(f"[Exploration] Could not use gripper (possibly missing predicate or blocked)")
        except Exception as e:
            print(f"[Exploration] Exception during gripper use: {e}")

        print("[Exploration] Exploration phase complete. Review the above logs to identify which predicate may be missing based on failed actions or exceptions.")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, update the domain/problem or inform the user.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()