# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try available skills to see which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills and observe the environment's response.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to use each skill on available objects/locations
        # and print out any errors or feedback to help identify the missing predicate.

        # Get all objects and locations from positions
        object_names = [name for name in positions.keys() if not name.startswith('location_')]
        location_names = [name for name in positions.keys() if name.startswith('location_')]

        # If locations are not named as 'location_*', try to infer from object positions
        if not location_names:
            # Try to extract unique locations from object positions
            location_set = set()
            for pos in positions.values():
                if isinstance(pos, dict) and 'location' in pos:
                    location_set.add(pos['location'])
            location_names = list(location_set)
            if not location_names:
                # Fallback: use some default location names
                location_names = ['ready-pose', 'drawer-area', 'bin-area']

        # Try to execute each skill in a safe, exploratory way
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_go':
                    # Try moving from one location to another
                    if len(location_names) >= 2:
                        from_loc = location_names[0]
                        to_loc = location_names[1]
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        print(f"Executed {skill} from {from_loc} to {to_loc}")
                elif skill == 'execute_pick':
                    # Try picking up an object from a location
                    if object_names and location_names:
                        obj = object_names[0]
                        loc = location_names[0]
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"Executed {skill} on {obj} at {loc}")
                elif skill == 'execute_place':
                    # Try placing an object into a location (e.g., drawer or bin)
                    if object_names and location_names:
                        obj = object_names[0]
                        loc = location_names[0]
                        obs, reward, done = execute_place(env, task, obj, loc)
                        print(f"Executed {skill} for {obj} to {loc}")
                elif skill == 'execute_push':
                    # Try pushing a drawer closed
                    if 'drawer' in positions:
                        drawer = 'drawer'
                    else:
                        drawer = None
                        for name in positions:
                            if 'drawer' in name:
                                drawer = name
                                break
                    if drawer and location_names:
                        loc = location_names[0]
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"Executed {skill} on {drawer} at {loc}")
                elif skill == 'execute_pull':
                    # Try pulling a drawer open using a handle
                    # Find a handle and a drawer
                    handle = None
                    drawer = None
                    for name in positions:
                        if 'handle' in name:
                            handle = name
                        if 'drawer' in name:
                            drawer = name
                    if handle and drawer and location_names:
                        loc = location_names[0]
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"Executed {skill} on {drawer} with {handle} at {loc}")
                elif skill == 'execute_sweep':
                    # Try sweeping an object on the floor
                    if object_names and location_names:
                        obj = object_names[0]
                        loc = location_names[0]
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"Executed {skill} on {obj} at {loc}")
                elif skill == 'execute_rotate':
                    # Try rotating an object (if applicable)
                    if object_names and location_names:
                        obj = object_names[0]
                        loc = location_names[0]
                        obs, reward, done = execute_rotate(env, task, obj, loc)
                        print(f"Executed {skill} on {obj} at {loc}")
                elif skill == 'execute_gripper':
                    # Try gripper action (no-op)
                    obs, reward, done = execute_gripper(env, task)
                    print(f"Executed {skill}")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === Main Task Plan (Placeholder) ===
        # After exploration, you would use the discovered information to construct the correct plan.
        # For now, this is left as a placeholder, as the main focus is on exploration for missing predicates.

        # TODO: After identifying the missing predicate, construct and execute the oracle plan using the available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
