# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (need-ready) ===
        # The feedback indicates that (not (need-ready)) is a missing predicate in preconditions.
        # We need to explore the environment to determine if 'need-ready' is required to proceed.
        # The plan is to attempt a skill that is blocked by (need-ready), and if blocked, execute the 'execute_go_ready' skill to clear it.

        # For demonstration, we will attempt to pick up an object from the floor.
        # If the action fails due to 'need-ready', we will execute 'execute_go_ready' and retry.

        # Find an object on the floor and its location
        object_on_floor = None
        object_pos = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('on_floor', False):
                object_on_floor = obj_name
                object_pos = obj_info.get('position', None)
                break

        # If no object found, just pick any object for demonstration
        if object_on_floor is None:
            for obj_name, obj_info in positions.items():
                object_on_floor = obj_name
                object_pos = obj_info.get('position', None)
                break

        # Assume we have a location called 'floor_location' for the object
        floor_location = None
        if object_pos is not None:
            floor_location = obj_info.get('location', None)
        if floor_location is None:
            floor_location = 'table'  # fallback

        # Try to execute a pick action, handle (need-ready) predicate
        print(f"[Exploration] Attempting to pick {object_on_floor} at {floor_location}")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=object_on_floor,
                location=floor_location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            # If the action fails due to 'need-ready', execute 'execute_go_ready' and retry
            print(f"[Exploration] Exception during pick: {e}")
            print("[Exploration] Executing execute_go_ready to clear 'need-ready' predicate.")
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=floor_location,
                    max_steps=50,
                    timeout=5.0
                )
                print("[Exploration] Retrying pick after execute_go_ready.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=object_on_floor,
                    location=floor_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e2:
                print(f"[Exploration] Failed to clear 'need-ready' or retry pick: {e2}")

        # === Main Task Plan (Example) ===
        # Here you would continue with the oracle plan, using only the available skills.
        # For demonstration, let's try a sequence: pick, place, push, pull, sweep, rotate, go, gripper

        # 1. Place the object (if holding)
        try:
            print(f"[Task] Attempting to place {object_on_floor} into a drawer (if available).")
            # Find a drawer in positions
            drawer_name = None
            for obj_name, obj_info in positions.items():
                if obj_info.get('type', '') == 'drawer':
                    drawer_name = obj_name
                    break
            if drawer_name is not None:
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj_name=object_on_floor,
                    drawer_name=drawer_name,
                    location=floor_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
        except Exception as e:
            print(f"[Task] Exception during place: {e}")

        # 2. Push the drawer closed (if available)
        try:
            if drawer_name is not None:
                print(f"[Task] Attempting to push {drawer_name} closed.")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawer_name,
                    location=floor_location,
                    max_steps=50,
                    timeout=5.0
                )
        except Exception as e:
            print(f"[Task] Exception during push: {e}")

        # 3. Pull the drawer open (if available)
        try:
            if drawer_name is not None:
                print(f"[Task] Attempting to pull {drawer_name} open.")
                # Find a handle for the drawer
                handle_name = None
                for obj_name, obj_info in positions.items():
                    if obj_info.get('type', '') == 'handle' and obj_info.get('drawer', '') == drawer_name:
                        handle_name = obj_name
                        break
                if handle_name is not None:
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer_name=drawer_name,
                        handle_name=handle_name,
                        location=floor_location,
                        max_steps=50,
                        timeout=5.0
                    )
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")

        # 4. Sweep action (if available)
        try:
            print(f"[Task] Attempting to sweep {object_on_floor}.")
            obs, reward, done = execute_sweep(
                env,
                task,
                obj_name=object_on_floor,
                location=floor_location,
                max_steps=50,
                timeout=5.0
            )
        except Exception as e:
            print(f"[Task] Exception during sweep: {e}")

        # 5. Rotate action (if available)
        try:
            print(f"[Task] Attempting to rotate {object_on_floor}.")
            obs, reward, done = execute_rotate(
                env,
                task,
                obj_name=object_on_floor,
                location=floor_location,
                max_steps=50,
                timeout=5.0
            )
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")

        # 6. Move (go) to another location (if available)
        try:
            print(f"[Task] Attempting to go from {floor_location} to 'ready-pose'.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=floor_location,
                to_location='ready-pose',
                max_steps=50,
                timeout=5.0
            )
        except Exception as e:
            print(f"[Task] Exception during go: {e}")

        # 7. Gripper action (if available)
        try:
            print(f"[Task] Attempting gripper action.")
            obs, reward, done = execute_gripper(
                env,
                task,
                max_steps=20,
                timeout=2.0
            )
        except Exception as e:
            print(f"[Task] Exception during gripper: {e}")

        print("[Task] Task execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()