# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the environment to determine the lock state of drawer1.
        # We use the available skills to do this.

        # Assumptions for object names and locations:
        # - The drawer is named 'drawer1'
        # - The handle is named 'handle1' and is associated with 'drawer1'
        # - The robot starts at 'ready-pose' or a known location
        # - The location of the handle and drawer are available in positions

        # You may need to adjust these names based on your environment's actual object naming
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        bin_name = 'bin'
        ready_pose = 'ready-pose'

        # Try to get locations; handle missing keys gracefully
        try:
            drawer_pos = positions[drawer_name]
        except KeyError:
            print(f"[Exploration] Could not find position for {drawer_name}.")
            drawer_pos = None
        try:
            handle_pos = positions[handle_name]
        except KeyError:
            print(f"[Exploration] Could not find position for {handle_name}.")
            handle_pos = None
        try:
            bin_pos = positions[bin_name]
        except KeyError:
            bin_pos = None

        # Step 1: Move to handle location if not already there
        # We'll assume the robot starts at 'ready-pose'
        current_location = ready_pose
        if handle_pos is not None:
            try:
                obs, reward, done = execute_go(env, task, current_location, handle_pos)
                current_location = handle_pos
            except Exception as e:
                print(f"[Exploration] Error moving to handle: {e}")

        # Step 2: Pick up the handle (if not already holding)
        try:
            obs, reward, done = execute_pick(env, task, handle_name, current_location)
        except Exception as e:
            print(f"[Exploration] Error picking handle: {e}")

        # Step 3: Move to drawer location (if different from handle)
        if drawer_pos is not None and current_location != drawer_pos:
            try:
                obs, reward, done = execute_go(env, task, current_location, drawer_pos)
                current_location = drawer_pos
            except Exception as e:
                print(f"[Exploration] Error moving to drawer: {e}")

        # Step 4: Try to pull the drawer to determine lock state
        # This will help us discover if (drawer-unlocked drawer1) is true or not
        try:
            obs, reward, done = execute_pull(env, task, drawer_name, handle_name, current_location)
            print("[Exploration] Pulled the drawer to check lock state.")
        except Exception as e:
            print(f"[Exploration] Error pulling drawer (may be locked): {e}")

        # At this point, the environment should have revealed the lock state of the drawer.
        # If the drawer was locked, the pull would fail or have no effect.
        # If unlocked, the drawer should now be open.

        # === Main Task Plan (After Exploration) ===
        # Now proceed with the main plan, assuming the missing predicate is now known.

        # Example: Pick an object from the floor and place it in the drawer (if open)
        # We'll use a generic object name 'object1' for demonstration
        object_name = 'object1'
        try:
            object_pos = positions[object_name]
        except KeyError:
            object_pos = None

        # Step 5: Move to object location
        if object_pos is not None and current_location != object_pos:
            try:
                obs, reward, done = execute_go(env, task, current_location, object_pos)
                current_location = object_pos
            except Exception as e:
                print(f"[Task] Error moving to object: {e}")

        # Step 6: Pick the object from the floor
        try:
            obs, reward, done = execute_pick(env, task, object_name, current_location)
        except Exception as e:
            print(f"[Task] Error picking object: {e}")

        # Step 7: Move to drawer location (if not already there)
        if drawer_pos is not None and current_location != drawer_pos:
            try:
                obs, reward, done = execute_go(env, task, current_location, drawer_pos)
                current_location = drawer_pos
            except Exception as e:
                print(f"[Task] Error moving to drawer for placing: {e}")

        # Step 8: Place the object into the drawer (if open)
        try:
            obs, reward, done = execute_place(env, task, object_name, drawer_name, current_location)
        except Exception as e:
            print(f"[Task] Error placing object in drawer: {e}")

        # Step 9: Optionally, push the drawer closed
        try:
            obs, reward, done = execute_push(env, task, drawer_name, current_location)
        except Exception as e:
            print(f"[Task] Error pushing drawer closed: {e}")

        # Step 10: Return to ready pose
        if current_location != ready_pose:
            try:
                obs, reward, done = execute_go(env, task, current_location, ready_pose)
                current_location = ready_pose
            except Exception as e:
                print(f"[Task] Error returning to ready pose: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()