# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (need-ready) ===
        # The feedback indicates that the predicate (need-ready) is missing or not handled.
        # In the domain, many actions set (need-ready) as an effect, and require (not (need-ready)) as a precondition.
        # Therefore, after any action that sets (need-ready), we must execute the "execute_go_ready" skill to clear it.

        # For demonstration, we will perform a minimal exploration:
        # 1. Try to pick an object (if available) from the floor.
        # 2. If the action sets (need-ready), attempt another action (e.g., pick or place) and handle failure.
        # 3. Then, execute "execute_go_ready" to clear (need-ready), and retry the action.

        # Find an object on the floor and its position
        object_on_floor = None
        object_pos = None
        for obj_name, pos in positions.items():
            if 'floor' in obj_name or 'obj' in obj_name or 'object' in obj_name:
                object_on_floor = obj_name
                object_pos = pos
                break

        # Fallback: just pick the first object if naming is not standard
        if object_on_floor is None and len(positions) > 0:
            object_on_floor = list(positions.keys())[0]
            object_pos = positions[object_on_floor]

        # Find a location for the robot to move to (if available)
        robot_location = None
        for name in positions:
            if 'ready' in name or 'pose' in name or 'start' in name:
                robot_location = name
                break
        if robot_location is None:
            robot_location = 'ready-pose'

        # 1. Move robot to the object's location (if not already there)
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=object_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {object_pos}")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # 2. Try to pick the object (this will set need-ready)
        need_ready_triggered = False
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj_name=object_on_floor,
                location=object_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Picked up {object_on_floor}")
            need_ready_triggered = True
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")
            need_ready_triggered = True  # Assume need-ready is set if pick fails

        # 3. Try to pick again (should fail if need-ready is set)
        if need_ready_triggered:
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=object_on_floor,
                    location=object_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] Unexpectedly succeeded in picking again (need-ready not enforced?)")
            except Exception as e:
                print("[Exploration] As expected, cannot pick again due to (need-ready):", e)

            # 4. Execute go_ready to clear (need-ready)
            try:
                obs, reward, done = execute_go_ready(
                    env,
                    task,
                    from_location=object_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] Executed go_ready to clear (need-ready)")
            except Exception as e:
                print(f"[Exploration] Error during execute_go_ready: {e}")

            # 5. Try to pick again (should now succeed)
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=object_on_floor,
                    location=object_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] Successfully picked again after go_ready (need-ready cleared)")
            except Exception as e:
                print("[Exploration] Still cannot pick after go_ready:", e)

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
